#!/usr/bin/env node

const readline = require('readline');

class TicTacToe {
  constructor() {
    this.board = [' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '];
    this.currentPlayer = 'X';
    this.gameOver = false;
    this.winner = null;
  }

  displayBoard() {
    console.log('\n');
    console.log(`  ${this.board[0]} | ${this.board[1]} | ${this.board[2]}`);
    console.log(' -----------');
    console.log(`  ${this.board[3]} | ${this.board[4]} | ${this.board[5]}`);
    console.log(' -----------');
    console.log(`  ${this.board[6]} | ${this.board[7]} | ${this.board[8]}`);
    console.log('\n');
  }

  displayPositions() {
    console.log('\nPosition numbers:');
    console.log('  1 | 2 | 3');
    console.log(' -----------');
    console.log('  4 | 5 | 6');
    console.log(' -----------');
    console.log('  7 | 8 | 9');
    console.log('');
  }

  makeMove(position) {
    const index = position - 1;

    if (position < 1 || position > 9) {
      return 'Invalid position! Choose 1-9.';
    }

    if (this.board[index] !== ' ') {
      return 'That position is already taken!';
    }

    this.board[index] = this.currentPlayer;

    if (this.checkWinner()) {
      this.gameOver = true;
      this.winner = this.currentPlayer;
      return `Player ${this.currentPlayer} wins!`;
    }

    if (this.checkDraw()) {
      this.gameOver = true;
      return "It's a draw!";
    }

    this.currentPlayer = this.currentPlayer === 'X' ? 'O' : 'X';
    return null;
  }

  checkWinner() {
    const winPatterns = [
      [0, 1, 2], [3, 4, 5], [6, 7, 8], // rows
      [0, 3, 6], [1, 4, 7], [2, 5, 8], // columns
      [0, 4, 8], [2, 4, 6]             // diagonals
    ];

    return winPatterns.some(pattern => {
      const [a, b, c] = pattern;
      return this.board[a] !== ' ' &&
             this.board[a] === this.board[b] &&
             this.board[a] === this.board[c];
    });
  }

  checkDraw() {
    return this.board.every(cell => cell !== ' ');
  }
}

async function playGame() {
  const game = new TicTacToe();
  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
  });

  const askQuestion = (query) => {
    return new Promise(resolve => rl.question(query, resolve));
  };

  console.log('\n=== TIC-TAC-TOE ===');
  game.displayPositions();

  while (!game.gameOver) {
    game.displayBoard();
    const position = await askQuestion(`Player ${game.currentPlayer}, enter position (1-9): `);
    const posNum = parseInt(position);

    const result = game.makeMove(posNum);
    if (result) {
      if (game.gameOver) {
        game.displayBoard();
        console.log(result);
      } else {
        console.log(result);
      }
    }
  }

  rl.close();
}

if (require.main === module) {
  playGame().catch(console.error);
}

module.exports = TicTacToe;
