//
//  ColumnHeaderController.m
//  Perforce
//
//  Created by Adam Czubernat on 19.06.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "ColumnViewHeader.h"

@interface ColumnViewHeader () <NSMenuDelegate> {
	
	P4Item *item;
	
	__weak IBOutlet PSView *selectedViewLeft;
	__weak IBOutlet PSView *selectedViewRight;

	__weak IBOutlet NSTextField *titleLabel;
	__weak IBOutlet PSView *imageView;
	
	__weak IBOutlet PSView *separatorView;
	__weak IBOutlet NSButton *favoriteButton;
	
	NSScrollView *_textview;
}
- (void)setItem:(P4Item *)anItem;
- (void)updateFavoritesNotification:(NSNotification *)notification;
- (IBAction)favoriteButtonPressed:(id)sender;
@end

@implementation ColumnViewHeader
@synthesize delegate, selected;

- (id)init {
	return self = [self initWithNibName:NSStringFromClass([self class]) bundle:nil];
}

- (void)dealloc {
	[[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (void)clicked {
	if ([delegate respondsToSelector:@selector(columnViewHeader:clickedWithItem:)])
		[delegate columnViewHeader:self clickedWithItem:item];
	[self setSelected:YES];
}

- (void)loadView {
	[super loadView];
	
	[(PSView *)self.view addTarget:self action:@selector(clicked) forEvents:PSViewEventClick];
	
	[[NSNotificationCenter defaultCenter]
	 addObserver:self selector:@selector(updateFavoritesNotification:)
	 name:P4WorkspaceDefaultsChangedNotification
	 object:nil];

	 // Load separator
	static NSImage *image;
	if (!image) {
		image = [NSImage imageNamed:@"SeparatorHorizontalDark.png"];
		image = [image resizableImageWithLeftCap:10.0f rightCap:1.0f];
	}
	[separatorView setContentMode:PSViewContentModeFillHorizontal];
	[separatorView setImage:image];
		
	// Assign colors
	NSColor *backgroundColor = [NSUserDefaults colorForKey:kColorBackgroundColumn];
	NSColor *leftColor = [NSUserDefaults colorForKey:kColorSideMenuCellPrimary];
	NSColor *rightColor = [NSUserDefaults colorForKey:kColorSideMenuCellSecondary];

	[(PSView *)self.view setBackgroundColor:backgroundColor];
	selectedViewLeft.backgroundColor = leftColor;
	selectedViewRight.backgroundColor = rightColor;
	
	[self setItem:item];
}

- (void)setRepresentedObject:(P4Item *)representedObject {
	[self setItem:representedObject];
}

#pragma mark - Public

- (void)setSelected:(BOOL)flag {
	
	selected = flag;
	
	static NSColor *textColor, *textColorSelected;
	if (!textColor) {
		textColor = [NSUserDefaults colorForKey:kColorTextColumn];
		textColorSelected = [NSUserDefaults colorForKey:kColorTextColumnSelected];
	}
		
	[selectedViewLeft setHidden:!selected];
	[selectedViewRight setHidden:!selected];
	
	titleLabel.textColor = selected ? textColorSelected : textColor;	
	imageView.image = selected ? item.iconHighlighted : item.icon;
	
	// Trim title label when showing favorite button
	CGRect frame = titleLabel.frame;
	frame.size.width = titleLabel.superview.frame.size.width - frame.origin.x;
	frame.size.width -= (selected ? 40.0f : 6.0f);
	titleLabel.frame = frame;
}

#pragma mark - Private

- (void)setItem:(P4Item *)anItem {
	item = anItem;
	titleLabel.stringValue = item.name ? : @"";

	NSString *image = item.isFavoriteFolder ? @"IconFavoriteSelected" : @"IconFavorite";
	[favoriteButton setImage:[NSImage imageNamed:image]];
	[self setSelected:selected];
}

- (void)updateFavoritesNotification:(NSNotification *)notification {
	NSString *image = item.isFavoriteFolder ? @"IconFavoriteSelected" : @"IconFavorite";
	[favoriteButton setImage:[NSImage imageNamed:image]];
}

#pragma mark Actions

- (IBAction)favoriteButtonPressed:(id)sender {
	if (item.isFavoriteFolder) {
		[item removeFavoriteFolder];
		[favoriteButton setImage:[NSImage imageNamed:@"IconFavorite"]];
	} else {
		[item addFavoriteFolder];
		[favoriteButton setImage:[NSImage imageNamed:@"IconFavoriteSelected"]];
	}	
}

#pragma mark - Menu delegate

- (void)menuWillOpen:(NSMenu *)menu {
	[menu setAllowsContextMenuPlugIns:NO];
	[menu setItem:item delegate:[self.view.window windowController]];
}

@end
