//
//  WorkspacePanelController.m
//  Perforce
//
//  Created by Adam Czubernat on 28.05.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "WorkspacePanelController.h"
#import "P4Workspace.h"

@interface WorkspacePanelController () {
	NSArray *workspaces;
	NSDictionary *selectedWorkspace;
	
	NSString *workspaceName;
	NSString *workspaceRoot;
	
	__weak IBOutlet NSView *selectionView;
	__weak IBOutlet NSView *creationView;

	// Workspace selection
	__weak IBOutlet NSPopUpButton *workspacePopUpButton;
	__weak IBOutlet NSButton *selectButton;
	
	__weak IBOutlet NSProgressIndicator *indicator;
	
	__weak IBOutlet NSTextField *statusLabel;

	__weak IBOutlet NSView *detailsView;
	__weak IBOutlet NSTextField *rootLabel;
	__weak IBOutlet NSTextField *depotLabel;
	__weak IBOutlet NSTextField *descriptionLabel;
	
	// Workspace creation
	__weak IBOutlet NSTextField *workspaceNameTextField;
	__weak IBOutlet NSTextField *workspaceRootTextField;
	__weak IBOutlet NSButton *createWorkspaceButton;
	
}
- (void)loadWorkspaces:(NSArray *)aWorkspaces;
- (IBAction)workspacePopUpChanged:(id)sender;
- (IBAction)cancelSelectionPressed:(id)sender;
- (IBAction)cancelCreationPressed:(id)sender;
- (IBAction)selectPressed:(id)sender;
- (IBAction)newWorkspacePressed:(id)sender;
- (IBAction)browsePressed:(id)sender;
- (IBAction)createPressed:(id)sender;
@end

@implementation WorkspacePanelController
@synthesize delegate;

- (void)windowDidLoad {
	[super windowDidLoad];	
	
	[workspacePopUpButton setAutoenablesItems:NO];
	[workspacePopUpButton removeAllItems];
	
	[detailsView setHidden:YES];
	[selectButton setEnabled:NO];
	rootLabel.stringValue =
	depotLabel.stringValue =
	descriptionLabel.stringValue = @"-";
	
	[self setPresentedView:selectionView];
	[indicator startAnimation:nil];

	// List workspaces	
	P4Workspace *workspace = [P4Workspace sharedInstance];
	[workspace listWorkspaces:^(P4Operation *operation, NSArray *response) {
		
		[indicator stopAnimation:nil];
		
		statusLabel.stringValue = @"Select workspace:";
		if (operation.errors) {
			[[NSAlert alertWithError:operation.error]
			 beginSheetModalForWindow:self.window
			 modalDelegate:nil
			 didEndSelector:nil
			 contextInfo:NULL];
		} else {

			[self loadWorkspaces:response];
			
			[detailsView setHidden:NO];
			[detailsView setAlphaValue:0.0f];
			[NSView animateWithDuration:0.25f animations:^{
				[detailsView.animator setAlphaValue:1.0f];
			}];
		}
	}];
}

#pragma mark - Public

#pragma mark - Private

- (void)loadWorkspaces:(NSArray *)response {

	workspaces = response;
	
	if (!workspaces.count)
		return;

	// Check workspaces existence on local filesystem

	NSString *lastWorkspace = [NSUserDefaults stringForKey:kDefaultWorkspace];
	NSMenuItem *selectedItem = nil;
	
	for (NSDictionary *dictionary in workspaces) {
		
		NSString *path = [dictionary objectForKey:@"Root"];
		NSString *name = [dictionary objectForKey:@"client"];
		
		BOOL isDirectory = NO;
		
		if ([[NSFileManager defaultManager]
			 fileExistsAtPath:path
			 isDirectory:&isDirectory] && isDirectory) {

			// Workspace folder exists, add as valid workspace
			[workspacePopUpButton addItemWithTitle:name];
			NSMenuItem *item = [workspacePopUpButton lastItem];
			[item setEnabled:YES];

			// Store first avaialble item or last used
			if (!selectedItem || [name isEqualToString:lastWorkspace])
				selectedItem = item;
			
		} else {
			
			[workspacePopUpButton addItemWithTitle:
			 [NSString stringWithFormat:@"%@ (unavailable)", name]];
			NSMenuItem *item = [workspacePopUpButton lastItem];
			[item setEnabled:NO];
		}
	}

	// Auto-select available item
	if (selectedItem) {
		[workspacePopUpButton selectItem:selectedItem];
		[selectButton setEnabled:YES];
	}
	
	[self workspacePopUpChanged:workspacePopUpButton];	
}

#pragma mark - Textfield delegate

- (void)controlTextDidChange:(NSNotification *)notification {
	workspaceRoot = workspaceRootTextField.stringValue;
	workspaceName = workspaceNameTextField.stringValue;
	
	BOOL directory;
	if ([[NSFileManager defaultManager]
		 fileExistsAtPath:workspaceRoot
		 isDirectory:&directory] && directory && workspaceName.length > 3) {
		
		[createWorkspaceButton setEnabled:YES];
		
	} else {
		
		[createWorkspaceButton setEnabled:NO];
	}
}

#pragma mark - Actions

- (IBAction)workspacePopUpChanged:(NSPopUpButton *)sender {
	NSUInteger index = [sender indexOfSelectedItem];
	selectedWorkspace = [workspaces objectAtIndex:index];
	rootLabel.stringValue = [selectedWorkspace objectForKey:@"Root"] ? : @"";
	depotLabel.stringValue = [selectedWorkspace objectForKey:@"Stream"] ? : @"";
	descriptionLabel.stringValue = [selectedWorkspace objectForKey:@"Description"] ? : @"";
}

- (IBAction)cancelSelectionPressed:(id)sender {
	[self dismiss];
	if ([delegate respondsToSelector:@selector(workspacePanelDidCancel)])
		[delegate workspacePanelDidCancel];
}

- (IBAction)cancelCreationPressed:(id)sender {
	[self setPresentedView:selectionView animated:YES];
}


- (IBAction)selectPressed:(NSButton *)sender {
	NSString *workspace = [selectedWorkspace valueForKey:@"client"];
	if (!workspace)
		return;
	
	[[P4Workspace sharedInstance] setWorkspace:workspace response:^(P4Operation *operation, NSArray *response) {
		if (operation.errors) {
			[[NSAlert alertWithError:operation.error]
			 beginSheetModalForWindow:self.window
			 modalDelegate:nil
			 didEndSelector:nil
			 contextInfo:NULL];
		} else {
			
			// Save last workspace details
			[NSUserDefaults setString:workspace forKey:kDefaultWorkspace];
			
			PSLogStore(@"Workspace info", @"%@", response);
			
			[self dismiss];
			
			if ([delegate respondsToSelector:@selector(workspacePanelDidSelectWorkspace:)])
				[delegate workspacePanelDidSelectWorkspace:workspace];
		}
	}];
}

- (void)newWorkspacePressed:(id)sender {
	[self setPresentedView:creationView animated:YES];
	[self.window makeFirstResponder:workspaceNameTextField];
}

- (void)browsePressed:(id)sender {
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];
	
    [openPanel setCanChooseFiles:NO];
    [openPanel setCanChooseDirectories:YES];
	[openPanel setAllowsMultipleSelection:NO];
	[openPanel setCanCreateDirectories:YES];
	[openPanel setPrompt:@"Select"];

	NSURL *directory = [NSURL fileURLWithPath:workspaceRoot ? : NSHomeDirectory()];
	[openPanel setDirectoryURL:directory];
	
	[openPanel beginSheetModalForWindow:self.window completionHandler:^(NSInteger result) {
		if (result != NSOKButton)
			return;
		workspaceRoot = [[(NSURL *)[[openPanel URLs] lastObject] path] directoryPath];
		workspaceRootTextField.stringValue = workspaceRoot ? : @"";
		[self controlTextDidChange:nil];
	}];
}

- (void)createPressed:(id)sender {
	
	if ([[NSAlert
		  alertWithMessageText:@"Please confirm"
		  defaultButton:@"Yes"
		  alternateButton:@"No"
		  otherButton:nil
		  informativeTextWithFormat:@"Do you want to create workspace named %@\n"
		  "in %@ directory?", workspaceName, workspaceRoot] runModal] != NSAlertDefaultReturn) {
		return;
	}
	
	NSDictionary *workspaceSpecs = @{
		@"Client" :	workspaceName,
		@"Owner" :	[[P4Workspace sharedInstance] username],
		@"Description" : @"Created by Perforce File Manager.",
		@"Root" :	workspaceRoot,
		@"Options" :	@"allwrite rmdir nocompress unlocked",
		@"SubmitOptions" : @"revertunchanged",
		@"LineEnd" : @"local",		 
	};
	
	[[P4Workspace sharedInstance]
	 createWorkspace:workspaceSpecs
	 response:^(P4Operation *operation, NSArray *response) {
		 if (operation.errors) {
			 [[NSAlert alertWithError:operation.error]
			  beginSheetModalForWindow:self.window
			  modalDelegate:nil
			  didEndSelector:nil
			  contextInfo:NULL];
			 return;
		 }
		 
		 [[P4Workspace sharedInstance]
		  setWorkspace:workspaceName
		  response:^(P4Operation *operation, NSArray *response) {
			  if (operation.errors) {
				  [[NSAlert alertWithError:operation.error]
				   beginSheetModalForWindow:self.window
				   modalDelegate:nil
				   didEndSelector:nil
				   contextInfo:NULL];
				  return;
			  }
			  [self dismiss];
			  if ([delegate respondsToSelector:@selector(workspacePanelDidSelectWorkspace:)])
				  [delegate workspacePanelDidCreateWorkspace:workspaceName];
		  }];
	 }];
}

@end
