//
//  P4Workspace.h
//  Perforce
//
//  Created by Adam Czubernat on 02/10/2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "P4Connection.h"
#import "P4NetworkOperation.h"

extern NSString * const P4ChangelistUpdatedNotification;
extern NSString * const P4UnreadUpdatedNotification;
extern NSString * const P4SyncStartedNotification;
extern NSString * const P4SyncFinishedNotification;
extern NSString * const P4SubmitFinishedNotification;

@protocol P4WorkspaceDelegate <NSObject>
// Filesystem Events
- (void)fileCreated:(NSString *)path;
- (void)fileRemoved:(NSString *)path;
- (void)fileMoved:(NSString *)path toPath:(NSString *)newPath;
- (void)fileRenamed:(NSString *)path toPath:(NSString *)newPath;
- (void)fileModified:(NSString *)path;
// P4 Events
- (void)file:(NSString *)path actionChanged:(NSDictionary *)info;
- (void)file:(NSString *)path revertedAction:(NSDictionary *)info;
- (void)file:(NSString *)path updated:(NSDictionary *)info;
- (void)file:(NSString *)path mappingChanged:(NSDictionary *)info;
- (void)file:(NSString *)path shelved:(NSDictionary *)info;
@end

// Credentials
@interface P4Credentials : NSObject
@property (nonatomic, retain) NSString *username;
@property (nonatomic, retain) NSString *password;
@property (nonatomic, retain) NSString *address;
@property (nonatomic, assign) BOOL resumeSession;
@end

@interface P4Workspace : NSObject

+ (P4Workspace *)sharedInstance;
- (void)addObserver:(id <P4WorkspaceDelegate>)observer;
- (void)removeObserver:(id <P4WorkspaceDelegate>)observer;

@property (readonly, nonatomic) NSString *address;
@property (readonly, nonatomic) NSString *username;
@property (readonly, nonatomic) NSString *workspace;
@property (readonly, nonatomic) NSString *root;

@property (readonly, nonatomic) NSString *syncDescription;

- (NSString *)ticket;

- (NSInteger)openedFilesCount;
- (NSInteger)addedFilesCount;
- (NSInteger)deletedFilesCount;
- (NSInteger)unreadCount;

// Login
- (BOOL)isConnected;
- (void)connectWithCredentials:(P4Credentials *)credentials response:(P4ResponseBlock_t)responseBlock;

- (BOOL)isLoggedIn;
- (void)loginWithCredentials:(P4Credentials *)credentials response:(P4ResponseBlock_t)responseBlock;
- (void)loginWithSSO:(P4Credentials *)credentials response:(P4ResponseBlock_t)responseBlock; // SSO
- (void)reloginWithCredentials:(P4Credentials *)credentials response:(P4ResponseBlock_t)responseBlock;
- (void)logout:(P4ResponseBlock_t)responseBlock;

- (void)listDepots:(P4ResponseBlock_t)responseBlock;
- (void)listWorkspaces:(P4ResponseBlock_t)responseBlock;

// Workspace
- (void)setWorkspace:(NSString *)workspace response:(P4ResponseBlock_t)responseBlock;
- (void)createWorkspace:(NSDictionary *)workspaceSpecs response:(P4ResponseBlock_t)responseBlock;

// Sync
- (BOOL)isSynchronizing;
- (void)syncWorkspace:(P4ReceiveBlock_t)receiveBlock response:(P4ResponseBlock_t)responseBlock;
- (void)setAutosyncInterval:(NSTimeInterval)interval; // Set non-zero to enable autosync
- (void)submitFiles:(NSArray *)paths message:(NSString *)message receive:(P4ReceiveBlock_t)receiveBlock response:(P4ResponseBlock_t)responseBlock;

// Mapping
- (NSArray *)mappingForPaths:(NSArray *)paths;
- (void)mappingSet:(BOOL)mapped path:(NSString *)path response:(P4ResponseBlock_t)responseBlock;
- (void)listMappings:(NSString *)path response:(P4ResponseBlock_t)responseBlock PS_DEPRECATED;

// Users
- (NSDictionary *)userInfo:(NSString *)username; // username or user@workspace

// Listing
- (void)listFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)listDepotFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)listPendingFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)listVersions:(NSString *)path response:(P4ResponseBlock_t)responseBlock;
- (void)listVersionsDetails:(NSArray *)versionPaths response:(P4ResponseBlock_t)responseBlock;

// File management
- (void)editFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)addFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)removeFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)moveFiles:(NSArray *)paths toPaths:(NSArray *)newPaths response:(P4ResponseBlock_t)responseBlock;
- (void)revertFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)revertUnchangedFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)reconcileFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)resolveFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)setAttribute:(NSString *)name value:(NSString *)value paths:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;

- (void)refreshMapping;
- (void)refreshPendingFiles;
- (void)refreshShelvedFiles;
- (void)refreshUsers;

// Shelving
- (NSArray *)shelvedForPaths:(NSArray *)paths;
- (void)shelveFile:(NSString *)path response:(P4ResponseBlock_t)responseBlock;
- (void)unshelveFile:(NSString *)path response:(P4ResponseBlock_t)responseBlock;
- (void)discardShelvedFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;
- (void)openShelvedFile:(NSString *)path response:(P4ResponseBlock_t)responseBlock;
- (void)listShelvedFiles:(NSArray *)paths response:(P4ResponseBlock_t)responseBlock;

// Searching
- (P4NetworkOperation *)searchFiles:(NSString *)search path:(NSString *)path response:(P4ResponseBlock_t)responseBlock;

// Unread
- (void)markFilesRead:(NSArray *)paths;
- (NSArray *)unreadForPaths:(NSArray *)paths;
- (void)listUnreadFiles:(NSString *)path response:(P4ResponseBlock_t)responseBlock;

// Generic
- (void)runCommand:(NSString *)command response:(P4ResponseBlock_t)responseBlock;

@end
