//
//  NSUserDefaults+Additions.m
//  Perforce
//
//  Created by Adam Czubernat on 13.06.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "NSUserDefaults+Additions.h"

@implementation NSUserDefaults (Additions)

static NSUserDefaults *defaults;

+ (void)loadDefaultsWithName:(NSString *)name {
	// Load default values
	NSString *path = [[NSBundle mainBundle] pathForResource:name ofType:@"plist"];
	NSDictionary *dictionary = [NSDictionary dictionaryWithContentsOfFile:path];
	[self loadDefaultsWithDictionary:dictionary];
}

+ (void)loadDefaultsWithDictionary:(NSDictionary *)dictionary {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	
	[defaults registerDefaults:dictionary];
	
	static NSString * const versionKey = @"Version";
	if (![defaults objectForKey:versionKey]) {
		// First-time launch store all values from defaults into system for editing
		for (NSString *key in dictionary)
			[defaults setObject:[dictionary objectForKey:key] forKey:key];
	}
	
	NSString *appVersion = [[NSBundle mainBundle] objectForInfoDictionaryKey:
							(NSString *)kCFBundleVersionKey];
	NSString *defaultsVersion = [defaults objectForKey:versionKey];
	if (![defaultsVersion isEqual:appVersion])
		[NSUserDefaults removeAllDefaults];
	
	[defaults setObject:appVersion forKey:versionKey];
	[defaults synchronize];
}

+ (void)removeAllDefaults {
	NSString *domainName = [[NSBundle mainBundle] bundleIdentifier];
	[[NSUserDefaults standardUserDefaults] removePersistentDomainForName:domainName];
}

+ (NSString *)stringForKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	return [defaults stringForKey:defaultName];
}

+ (void)setString:(NSString *)value forKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	[defaults setObject:value forKey:defaultName];
	[defaults synchronize];
}

+ (NSColor *)colorForKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];

	static NSMutableDictionary *colorCache;
	if (!colorCache)
		colorCache = [NSMutableDictionary dictionary];
	
	NSString *colorName = [defaults stringForKey:defaultName];
	NSColor *color = [colorCache objectForKey:colorName];
	if (!color) {
		color = [NSColor colorWithHexString:colorName];
		if (color)
			[colorCache setObject:color forKey:colorName];
	}
	
	return color;
}

+ (NSInteger)integerForKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	return [defaults integerForKey:defaultName];
}

+ (CGFloat)floatForKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	return [defaults floatForKey:defaultName];
}

+ (BOOL)boolForKey:(NSString *)defaultName {
	if (!defaults)
		defaults = [NSUserDefaults standardUserDefaults];
	return [defaults boolForKey:defaultName];
}

@end
