//
//  P4ChangelistItem.m
//  Perforce
//
//  Created by Adam Czubernat on 10.07.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "P4ChangelistItem.h"

@interface P4ChangelistItem () {
	NSPredicate *predicate;
}
- (id)initWithDictionary:(NSDictionary *)dictionary parentItem:(P4Item *)parent;
@end

@implementation P4ChangelistItem

#pragma mark - P4Item Override

- (id)init {
	self = [super init];
	if (self) {
		flags.directory = YES;
		name = @"Changelist";
		localPath = @"changelist://";
		[[P4Workspace sharedInstance] addObserver:self];
	}
	return self;
}

- (id)defaultAction {
	if (!flags.tracked)
		return nil;
	
	return [P4ItemAction
			actionForItem:self name:@"Open"
			selector:@selector(open)];
}

- (NSArray *)actions {
	if (flags.directory)
		return nil;	
	NSMutableArray *actions = (NSMutableArray *)[super actions];
	return actions;
}

- (NSArray *)actionsForItems:(NSArray *)items {
	NSMutableArray *actions = (NSMutableArray *)[super actionsForItems:items];
	[actions addObject:[P4ItemAction
						actionForItems:items name:@"Check-in selected files"
						selector:@selector(checkInItems:)]];
	[actions addObject:[P4ItemAction
						actionForItems:items name:@"Revert selected files"
						selector:@selector(revertItems:)]];
	return actions;
}

- (void)loadPath:(NSString *)path {
	
	children = nil;
	flags.loading = YES;
	
	[[P4Workspace sharedInstance]
	 listPendingFiles:nil
	 response:^(P4Operation *operation, NSArray *response) {
		 if (operation.errors) {
			 [self failWithError:operation.error];
			 return;
		 }
		 
		 // Serialize children from response
		 NSMutableArray *array = [NSMutableArray arrayWithCapacity:512];
		 for (NSDictionary *childDict in response) {
			 P4ChangelistItem *child = [[P4ChangelistItem alloc]
										initWithDictionary:childDict
										parentItem:self];
			 
			 // Filter using predicate
			 if (!predicate || [predicate evaluateWithObject:child])
				 [array addObject:child];
		 }
		 
		 // Sort files alphanumerically
		 children = array;
		 [self sortChildren];
		 
		 [self finishLoading];
	 }];
}

#pragma mark - Public

- (void)setType:(NSString *)type {
	localPath = [NSString stringWithFormat:@"changelist://%@", type];
}

- (void)setPredicate:(NSPredicate *)aPredicate {
	predicate = aPredicate;
}

#pragma mark - Private

- (id)initWithDictionary:(NSDictionary *)dictionary parentItem:(P4Item *)parentItem {
	if (self = [super init]) {
		P4Workspace *p4 = [P4Workspace sharedInstance];
		
		parent = parentItem;
		metadata = dictionary;
		
		remotePath = [dictionary objectForKey:@"depotFile"];
		name = remotePath.lastPathComponent;
		
		// Create local path
		localPath = [dictionary objectForKey:@"clientFile"];
		NSString *client = [dictionary objectForKey:@"client"] ?: @"";
		NSString *clientPrefix = [NSString stringWithFormat:@"//%@/", client];
		if ([localPath hasPrefix:clientPrefix]) {
			localPath = [localPath substringFromIndex:clientPrefix.length];
			localPath = [[p4 root] stringByAppendingPath:localPath];
		}
				
		// Set metadata
		status = [metadata objectForKey:@"action"];
		NSString *user = [metadata objectForKey:@"otherOpen0"];
		NSDictionary *info = [p4 userInfo:user];
		statusOwner = [info objectForKey:@"Email"] ?: user;
		flags.tracked = YES; // Always tracked
		[self refreshTags];
		
		// Shelved and unread
		if (remotePath)
			flags.shelved = [p4 shelvedForPaths:@[ remotePath ]].count > 0;
		if (localPath)
			flags.unread = [p4 unreadForPaths:@[ localPath ]].count > 0;
	}
	return self;
}

#pragma mark P4Workspace Events

- (void)file:(NSString *)filePath actionChanged:(NSDictionary *)info {
	
	if (!self->children)
		return; // Not loaded
	
	NSString *action = [info objectForKey:@"action"];
	
	P4Item *item = [self cachedItemForPath:filePath];
	if ([item.status isEqualToString:action])
		return; // Nothing changed
	
	if (!item && !action)
		return; // No need to add
	
	if (!item) {
		// Update children by adding new item
		P4Item *newItem = [[P4ChangelistItem alloc]
						   initWithDictionary:info
						   parentItem:self];
		// Add item to parent
		NSMutableArray *array = [NSMutableArray arrayWithArray:self->children];
		// Filter using predicate
		if (!predicate || [predicate evaluateWithObject:newItem])
			[array addObject:newItem];
		
		// Sort files alphanumerically
		self->children = array;
		[self sortChildren];
		[self finishLoading];
		
	} else {
		item->status = action;
		[item finishLoading];
	}
}

- (void)file:(NSString *)filePath revertedAction:(NSDictionary *)info {
	
	if (!self->children)
		return; // Not loaded

	P4Item *item = [self cachedItemForPath:filePath];
	if (!item)
		return; // Nothing changed
	
	NSMutableArray *array = [NSMutableArray arrayWithArray:self->children];
	[array removeObject:item];
	self->children = array;
	
	[self finishLoading];
	[item invalidate];
}

@end
