//
//  PSPathComponentCell.m
//  Perforce
//
//  Created by Adam Czubernat on 17.06.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "PSPathComponentCell.h"

@implementation PSPathComponentCell
@synthesize lastItem, backgroundColor;

#pragma mark - Overrides

- (NSSize)cellSizeForBounds:(NSRect)aRect {
	CGSize size = [super cellSizeForBounds:aRect];
	if (CGRectIsEmpty(aRect))
		size.width = [self minimumSize];
	else
		size.width = [self fullSize];
	
	return size;
}

- (NSRect)imageRectForBounds:(NSRect)frame {
	CGSize size = self.imageSize;
	frame.origin.x += self.spacing;
	frame.origin.y = truncf((frame.size.height - size.height) * 0.5f);
	frame.size = size;
	return frame;
}

- (NSRect)titleRectForBounds:(NSRect)frame {
	CGFloat spacing = self.spacing;
	CGSize size = self.titleSize;
	frame.origin.x += self.imageSize.width + 2.0f * spacing;
	frame.origin.y = truncf((frame.size.height - size.height) * 0.5f);
	frame.size.width = size.width + spacing - (self.fullSize - frame.size.width);
	return frame;
}

- (void)drawWithFrame:(CGRect)cellFrame inView:(NSView *)controlView {
	
	// Clear background
	[backgroundColor set];
	NSRectFill(cellFrame);
	
	CGRect titleFrame = [self titleRectForBounds:cellFrame];
	CGFloat spacing = self.spacing;
	
	// Draw image
	[self.image
	 drawInRect:[self imageRectForBounds:cellFrame]
	 fromRect:CGRectZero
	 operation:NSCompositeSourceOver
	 fraction:1.0f
	 respectFlipped:YES
	 hints:nil];
			
	if (titleFrame.size.width < 0.0f)
		return;
	
	// Draw title
	[self.title
	 drawInRect:titleFrame
	 withAttributes:[[self attributedStringValue] attributesAtIndex:0 effectiveRange:0]];
	
	// Draw title fade effect
	NSGradient *fade = [[NSGradient alloc]
						initWithStartingColor:[backgroundColor colorWithAlphaComponent:0.0f]
						endingColor:backgroundColor];
	[fade drawInRect:(CGRect) {
		titleFrame.origin.x + titleFrame.size.width - spacing, 0.0f,
		spacing, cellFrame.size.height,
	} angle:0.0f];

	// Don't draw separator behind last item
	if (lastItem)
		return;
	
	// Draw separator
	NSImage *separator = self.separatorImage;
	CGSize separatorSize = separator.size;
	[separator
	 drawInRect:(CGRect) {
		 titleFrame.origin.x + titleFrame.size.width,
		 truncf((cellFrame.size.height - separatorSize.height) * 0.5f),
		 separatorSize,
	 }
	 fromRect:NSZeroRect
	 operation:NSCompositeSourceOver
	 fraction:1.0f
	 respectFlipped:YES
	 hints:nil];
}

#pragma mark - Public

- (CGSize)titleSize {
	return [[self attributedStringValue] size];
}

#pragma mark - Template

- (NSImage *)separatorImage {
	static NSImage *image;
	return image ? : (image = [NSImage imageNamed:@"PathControlArrow.png"]);
}

- (CGFloat)fullSize {
	return self.minimumSize + self.titleSize.width + self.spacing;
}

- (CGFloat)minimumSize {
	CGFloat size = self.imageSize.width + 2 * self.spacing;
	if (!lastItem)	// Add separator on intermediate items
		size += self.separatorImage.size.width;
	return size;
}

- (CGSize)imageSize {
	return (CGSize) { 32.0f, 32.0f };
}

- (CGFloat)spacing {
	return 10.0f;
}
		 
@end
