//
//  DebugWindowController.m
//  Perforce
//
//  Created by Adam Czubernat on 20.05.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "DebugWindowController.h"
#import "P4Workspace.h"

@interface DebugWindowController () <NSWindowDelegate> {
	__weak IBOutlet NSTextField *commandTextField;
	__unsafe_unretained IBOutlet NSTextView *resultTextView;
}
- (void)appendLog:(NSString *)log;
- (void)debugLogNotification:(NSNotification *)notification;
- (IBAction)runPressed:(id)sender;
- (IBAction)savePressed:(id)sender;
- (IBAction)clearPressed:(id)sender;
@end

@implementation DebugWindowController

- (id)init {
	return self = [super initWithWindowNibName:NSStringFromClass([self class])];
}

- (void)showWindow:(id)sender {
	[super showWindow:sender];
	[self.window setDelegate:self];
	
	[[NSNotificationCenter defaultCenter]
	 addObserver:self
	 selector:@selector(debugLogNotification:)
	 name:PSDebugLogNotification
	 object:nil];
	
	NSArray *log = PSDebugLogDump();
	
	for (NSString *string in log) {
		[self appendLog:string];
	}
	[resultTextView scrollToEndOfDocument:nil];
}

- (void)appendLog:(NSString *)log {
	static NSAttributedString *newline;
	if (!newline)
		newline = [[NSAttributedString alloc] initWithString:@"\n"];
	static NSDictionary *errorAttr, *infoAttr;
	if (!errorAttr) {
		errorAttr = @{ NSForegroundColorAttributeName : [NSColor redColor] };
		infoAttr = @{ NSForegroundColorAttributeName : [NSColor grayColor] };
	}
	
	NSDictionary *attr = nil;
	if ([log hasPrefix:@"Error >"])
		attr = errorAttr;
	else if ([log hasPrefix:@"P4 >"])
		attr = infoAttr;
	
	NSAttributedString *string = [[NSAttributedString alloc]
								  initWithString:log
								  attributes:attr];
	[[resultTextView textStorage] appendAttributedString:string];
	[[resultTextView textStorage] appendAttributedString:newline];
}

#pragma mark - NSWindowDelegate

- (void)windowWillClose:(NSNotification *)notification {
	[[NSNotificationCenter defaultCenter]
	 removeObserver:self
	 name:PSDebugLogNotification
	 object:nil];
}

#pragma mark PSDebugLogNotification

- (void)debugLogNotification:(NSNotification *)notification {
	NSString *log = notification.object;
	if (!log) {
		[resultTextView setString:@""];
		return;
	}
	[self appendLog:log];
	[resultTextView scrollRangeToVisible:
	 NSMakeRange([[resultTextView string] length], 0)];
}

#pragma mark - Actions

- (IBAction)runPressed:(id)sender {
	
	if (!commandTextField.stringValue.length)
		return;
	
	P4Workspace *workspace = [P4Workspace sharedInstance];
	
	if (![workspace isLoggedIn]) {
		[self appendLog:@"Error > Not connected"];
		return;
	}	
	
	[workspace runCommand:commandTextField.stringValue response:^(P4Operation *operation, NSArray *response) {
			
		if (!response)
			PSLog(@"Response: NULL");
		else
			PSLog(@"Response:\n%@\nRecords: %ld", response, response.count);
	}];
}

- (IBAction)savePressed:(id)sender {
	PSDebugLogSave();
}

- (IBAction)clearPressed:(id)sender {
	PSDebugLogClear();
}

@end
