//
//  P4Item.h
//  Perforce
//
//  Created by Adam Czubernat on 31.05.2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "P4Workspace.h"
#import "P4WorkspaceDefaults.h"

@protocol P4ItemDelegate, P4ItemActionDelegate;

@interface P4Item : NSObject <P4WorkspaceDelegate, QLPreviewItem> {
	__unsafe_unretained P4Item *parent;
	NSString *name, *localPath, *remotePath, *status, *statusOwner;
	NSArray *children;
	NSDictionary *metadata;
	struct {
		unsigned int directory:1;
		unsigned int loading:1;
		unsigned int failure:1;
		unsigned int tracked:1;
		unsigned int hasMapped:1;
		unsigned int mapped:1;
		unsigned int ignored:1;
		unsigned int unread:1;
		unsigned int shelved:1;
	} flags;
}

- (P4Item *)parent;
- (NSString *)name;
- (NSString *)path;
- (NSString *)localPath;
- (NSString *)remotePath;

- (BOOL)isDirectory;
- (NSArray *)children;

- (BOOL)isLoading;
- (BOOL)hasError;

- (BOOL)isTracked;
- (BOOL)hasMapped;
- (BOOL)isMapped;
- (BOOL)isIgnored;
- (BOOL)isUnread;
- (BOOL)isShelved;
- (NSString *)status;
- (NSString *)statusOwner;
- (NSArray *)tags;
- (BOOL)hasTag:(NSString *)tag; // Case insensitive

- (NSDictionary *)metadata;
- (NSImage *)icon;
- (NSImage *)iconHighlighted;
- (NSColor *)overlay;
- (NSImage *)previewWithSize:(CGSize)size;

// Actions
- (void)open;
- (void)openFromDepot;
- (void)openWithCheckout;
- (void)showInFinder;
- (void)checkout;
- (void)checkoutItems:(NSArray *)items;
- (void)addItem;
- (void)addItems:(NSArray *)items;
- (void)checkIn;
- (void)checkInItems:(NSArray *)items;
- (void)checkInAll;
- (void)deleteItem;
- (void)deleteItems:(NSArray *)items;
- (void)revert;
- (void)revertItems:(NSArray *)items;
- (void)revertIfUnchanged;
- (void)mapToWorkspace;
- (void)unmapFromWorkspace;
- (void)markAsRead;
- (void)markItemsAsRead:(NSArray *)items;
- (void)markAllAsRead;
- (void)shelve;
- (void)unshelve;
- (void)discardShelve;
- (void)openShelve;
- (void)showVersions;
- (void)copyShareLink;

- (void)openVersion:(NSString *)versionPath;
- (void)revertToVersion:(NSString *)versionPath;

// Favorite folders
- (BOOL)isFavoriteFolder;
- (void)addFavoriteFolder;
- (void)removeFavoriteFolder;

// Edit actions
- (BOOL)isEditable;
- (void)createDirectory;
- (void)rename:(NSString *)name;
- (void)addFiles:(NSArray *)paths;
- (void)copyFiles:(NSArray *)paths;
- (void)moveFiles:(NSArray *)paths;
- (void)addTag:(NSString *)tag;
- (void)removeTag:(NSString *)tag;

// Contextual actions
- (id)defaultAction; // Override
- (NSArray *)actions; // Override
- (NSArray *)actionsForItems:(NSArray *)items; // Override
- (void)performAction:(SEL)selector object:(id)object delegate:(id)delegate;
- (void)performAction:(SEL)selector items:(NSArray *)items delegate:(id)delegate;

// Utils
+ (void)addObserver:(id <P4ItemDelegate>)observer;
+ (void)removeObserver:(id <P4ItemDelegate>)observer;

- (void)loadPath:(NSString *)path; // Override
- (P4Item *)cachedItemForPath:(NSString *)path;
- (void)reload;

// Protected
- (void)sortChildren;
- (void)finishLoading;
- (void)failWithError:(NSError *)error;
- (void)invalidate;
- (void)refreshTags;

// Filesystem Events
- (void)fileCreated:(NSString *)path;
- (void)fileRemoved:(NSString *)path;
- (void)fileMoved:(NSString *)path toPath:(NSString *)newPath;
- (void)fileRenamed:(NSString *)path toPath:(NSString *)newPath;
- (void)fileModified:(NSString *)path;
// P4 Events
- (void)file:(NSString *)path actionChanged:(NSDictionary *)info;
- (void)file:(NSString *)path revertedAction:(NSDictionary *)info;
- (void)file:(NSString *)path updated:(NSDictionary *)info;
- (void)file:(NSString *)path mappingChanged:(NSDictionary *)info;
- (void)file:(NSString *)path shelved:(NSDictionary *)info;
@end

// P4Item Delegate

@protocol P4ItemDelegate <NSObject>
- (void)itemDidLoad:(id)item;			// Loaded children list, renamed, modified
- (void)itemDidInvalidate:(id)item;		// Deleted or moved outside parent
- (void)item:(id)item didFailWithError:(NSError *)error;
@end

// P4ItemAction

@interface P4ItemAction : NSObject
+ (id)actionForItem:(P4Item *)item name:(NSString *)name selector:(SEL)selector;
+ (id)actionForItems:(NSArray *)items name:(NSString *)name selector:(SEL)selector;
@property (nonatomic, unsafe_unretained) id <P4ItemActionDelegate> delegate;
@property (nonatomic, unsafe_unretained) P4Item *item;
@property (nonatomic, retain) NSArray *items;
@property (nonatomic, retain) NSString *name;
@property (nonatomic, assign) SEL selector;
@property (nonatomic, retain) id object;
@property (nonatomic, assign) BOOL disabled;
- (void)performAction;
@end

// P4ItemAction Delegate

@protocol P4ItemActionDelegate <NSObject>
- (void)action:(id)action loadingMessage:(NSString *)message;
- (void)action:(id)action didFinish:(NSArray *)response error:(NSError *)error;
- (BOOL)action:(id)action prompt:(NSString *)message;
@end



