//
//  SidebarViewCell.m
//  Perforce
//
//  Created by Adam Czubernat on 16/12/2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "SidebarViewCell.h"
#import <objc/objc-runtime.h>

@interface SidebarViewCell () <NSTextFieldDelegate> {
	
	__unsafe_unretained id target;
	SEL action;
		
	BOOL editable;
	BOOL selectable;	
	NSImage *indicatorImage;
	
	__weak IBOutlet PSView *leftView;
	__weak IBOutlet PSView *rightView;
	
	__weak IBOutlet NSButton *indicatorButton;
	__weak IBOutlet NSTextField *indicatorTextField;
	__weak IBOutlet NSImageView *indicatorImageView;
}
- (void)editAction;
- (void)setHover:(BOOL)hovered selected:(BOOL)selected;
@end

@implementation SidebarViewCell
@synthesize indicatorText, editing, selected;

#pragma mark - Overrides

- (void)awakeFromNib {	
    [self addTrackingArea:
	 [[NSTrackingArea alloc]
	  initWithRect:NSZeroRect
	  options:NSTrackingActiveInKeyWindow | NSTrackingMouseEnteredAndExited |
	  NSTrackingAssumeInside | NSTrackingInVisibleRect
	  owner:self
	  userInfo:nil]];
	
	[indicatorButton setHidden:YES];
	[indicatorTextField setHidden:YES];

	[indicatorButton setTarget:self];
	[indicatorButton setAction:@selector(editAction)];
	
	[self.textField setDelegate:self];
}

- (void)mouseEntered:(NSEvent *)theEvent {
	if (self.backgroundStyle == NSBackgroundStyleLight)
		[self setHover:YES selected:NO];
}

- (void)mouseExited:(NSEvent *)theEvent {
	if (self.backgroundStyle == NSBackgroundStyleLight)
		[self setHover:NO selected:NO];
}

- (void)setBackgroundStyle:(NSBackgroundStyle)backgroundStyle {
	[super setBackgroundStyle:backgroundStyle];
	BOOL highlighted = backgroundStyle != NSBackgroundStyleLight;
	[self setHover:highlighted selected:highlighted];
}

#pragma mark - Public 

- (void)makeIndicating {
	editable = NO;
	selected = selectable = NO;
	[self.textField setEditable:NO];
	[self setIndicatorText:nil];
	indicatorImage = [NSImage imageNamed:@"ButtonRight"];
}

- (void)makeEditable {
	editable = YES;
	selected = selectable = NO;
	[self.textField setEditable:YES];
	[self setIndicatorText:nil];
	indicatorImage = [NSImage imageNamed:@"ButtonEdit"];
}

- (void)makeSelectable {
	selectable = YES;
	editable = NO;
	[self.textField setEditable:NO];
	[self setIndicatorText:nil];
	indicatorImage = [NSImage imageNamed:@"ButtonRight"];
	selected = NO;
}

- (void)setIndicatorText:(NSString *)text {
	BOOL hidden = !(indicatorText = text).length;
	indicatorTextField.stringValue = text ?: @"";
	[indicatorTextField setHidden:hidden];
	[indicatorImageView setHidden:!hidden];
}

- (void)setTarget:(id)aTarget action:(SEL)anAction {
	target = aTarget;
	action = anAction;	
	[indicatorButton setHidden:!(target && action)];
}

#pragma mark - Private

- (void)editAction {
	[self.window makeFirstResponder:self.textField];
	editing = YES;
	[self setHover:NO selected:NO];
}

- (void)setHover:(BOOL)hovered selected:(BOOL)highlighted {
	
	static NSColor *leftColor, *rightColor, *leftColorHovered, *rightColorHovered;
	static NSColor *textColor, *textColorHovered, *textColorIndicator;
	static NSImage *selectedImage, *selectedImageHover;
	// Load images and colors
	static dispatch_once_t onceToken;
	dispatch_once(&onceToken, ^{
		leftColor = [NSUserDefaults colorForKey:kColorSideMenuCellPrimary];
		leftColorHovered = [NSUserDefaults colorForKey:kColorSideMenuCellHoveredPrimary];
		rightColor = [NSUserDefaults colorForKey:kColorSideMenuCellSecondary];
		rightColorHovered = [NSUserDefaults colorForKey:kColorSideMenuCellHoveredSecondary];
		textColor = [NSUserDefaults colorForKey:kColorTextSideMenu];
		textColorHovered = [NSUserDefaults colorForKey:kColorTextSideMenuSelected];
		textColorIndicator = [NSColor grayColor];
		selectedImage = [NSImage imageNamed:@"IconCheckmarkDark"];
		selectedImageHover = [NSImage imageNamed:@"IconCheckmark"];
	});
	
	if (editing)
		highlighted = hovered = NO;
		
	if (hovered) {
		leftView.backgroundColor = highlighted ? leftColor : leftColorHovered;
		rightView.backgroundColor = highlighted ? rightColor : rightColorHovered;
	}
	
	[leftView setHidden:!hovered];
	[rightView setHidden:!hovered];

	NSImage *image = indicatorImage;
	BOOL hidden = !hovered || indicatorText.length;
	
	if (selectable && selected) {
		hidden = NO;
		image = hovered ? selectedImageHover : selectedImage;
	}
	
	[indicatorImageView setImage:image];
	[indicatorImageView setHidden:hidden];
	
	// Trim titlelabel when showing indicator
	CGRect frame = self.textField.frame;
	frame.size.width = self.bounds.size.width - (hovered ? 40.0f : 6.0f) - frame.origin.x;
	self.textField.frame = frame;
	
	// Colors
	self.textField.textColor = hovered ? textColorHovered : textColor;
	indicatorTextField.textColor = hovered ? textColorHovered : textColorIndicator;
}

#pragma mark - TextField delegate

- (void)cancelOperation:(id)sender {
	editing = NO;
	[self.textField abortEditing];
}

- (void)controlTextDidEndEditing:(NSNotification *)obj {
	editing = NO;
	if (target && action)
		objc_msgSend(target, action, self);
}

@end
