//
//  VersionsViewCell.m
//  Perforce
//
//  Created by Adam Czubernat on 19/12/2013.
//  Copyright (c) 2013 Perforce Software, Inc. All rights reserved.
//

#import "VersionsViewCell.h"
#import "TagsView.h"

@interface VersionsViewCell () {
	
	CGFloat rowHeight;

	// Outlets
	__weak IBOutlet NSBox *leftBox;
	__weak IBOutlet NSBox *contentBox;
	__weak IBOutlet NSTextField *versionLabel;
	__weak IBOutlet NSBox *titleSeparator;
	__weak IBOutlet NSButton *firstButton;
	__weak IBOutlet NSButton *secondButton;
	__weak IBOutlet NSButton *thirdButton;
	__weak IBOutlet NSTextField *titleLabel;
	__weak IBOutlet NSTextField *authorDetailTitle;
	__weak IBOutlet NSTextField *authorDetailLabel;
	__weak IBOutlet NSTextField *dateDetailTitle;
	__weak IBOutlet NSTextField *dateDetailLabel;
	__weak IBOutlet NSTextField *actionDetailTitle;
	__weak IBOutlet NSTextField *actionDetailLabel;
	__weak IBOutlet NSTextField *tagsTitle;
	__weak IBOutlet TagsView *tagsView;	
}
- (void)setVersionNumber:(NSString *)version;
- (void)setDate:(NSNumber *)date;
@end

@implementation VersionsViewCell
@synthesize showsButtons, statusColor;

- (void)awakeFromNib {
	[super awakeFromNib];
	[tagsView setAllowsAdding:NO];
	
	[self setTags:nil];
}

- (void)mouseDown:(NSEvent *)theEvent {
	// Hack in order to pass drag events into tagsview when clicking on tagsview subviews
	CGPoint point = [tagsView.superview convertPoint:theEvent.locationInWindow fromView:nil];
	NSView *view = [tagsView hitTest:point];
	if (view && view != tagsView)
		return;
	[super mouseDown:theEvent];
}

- (void)setBackgroundStyle:(NSBackgroundStyle)backgroundStyle {
	if (backgroundStyle == self.backgroundStyle)
		return;
	
	static NSColor *colorText, *colorTextSelected;
	static NSColor *colorFill, *colorFillSelected;
	// Load images and colors
	static dispatch_once_t onceToken;
	dispatch_once(&onceToken, ^{
		colorText = [NSUserDefaults colorForKey:kColorTextColumn];
		colorTextSelected = [NSUserDefaults colorForKey:kColorTextColumnSelected];
		colorFill = [NSUserDefaults colorForKey:kColorBackgroundIcon];
		colorFillSelected = [NSUserDefaults colorForKey:kColorBackgroundSelected];
	});
	
	[super setBackgroundStyle:backgroundStyle];
	BOOL selected = backgroundStyle != NSBackgroundStyleLight;
	
	NSColor *fillColor = selected ? colorFillSelected : colorFill;
	NSColor *textColor = selected ? colorTextSelected : colorText;
	
	[contentBox setFillColor:fillColor];
	[versionLabel setTextColor:textColor];
	[titleSeparator setFillColor:selected ? textColor : colorFillSelected];
	[titleLabel setTextColor:textColor];
	[authorDetailTitle setTextColor:textColor];
	[authorDetailLabel setTextColor:textColor];
	[dateDetailTitle setTextColor:textColor];
	[dateDetailLabel setTextColor:textColor];
	[actionDetailTitle setTextColor:textColor];
	[actionDetailLabel setTextColor:textColor];
	[tagsTitle setTextColor:textColor];
	
	[firstButton setHidden:!selected || !showsButtons];
	[secondButton setHidden:!selected || !showsButtons];
	[thirdButton setHidden:!selected || !showsButtons];
	
	CGRect frame = titleLabel.frame;
	frame.size.width = (selected && showsButtons ? firstButton.frame.origin.x :
						self.frame.size.width - 12.0f) - frame.origin.x;
	titleLabel.frame = frame;
}

#pragma mark - Private

- (void)setVersionNumber:(NSString *)version {
	CGRect frame;
	CGFloat delta;
	
	delta = versionLabel.frame.size.width;
	versionLabel.stringValue = [NSString stringWithFormat:@"V.%@", version];
	[versionLabel sizeToFit];
	delta = versionLabel.frame.size.width - delta;
	
	frame = titleSeparator.frame;
	frame.origin.x += delta;
	titleSeparator.frame = frame;
	
	frame = titleLabel.frame;
	frame.size.width -= delta;
	frame.origin.x += delta;
	titleLabel.frame = frame;
}

- (void)setDate:(NSNumber *)time {
	static NSDateFormatter *formatter;
	if (!formatter) {
		formatter = [[NSDateFormatter alloc] init];
		formatter.dateStyle = NSDateFormatterLongStyle;
	}
	NSDate *date = time ? [NSDate dateWithTimeIntervalSince1970:time.doubleValue] : nil;
	dateDetailTitle.stringValue = @"CHECKED IN";
	dateDetailLabel.stringValue = [formatter stringFromDate:date] ?: @"";
}

#pragma mark - Public

- (CGFloat)rowHeight {
	return rowHeight;
}

- (void)setButtonsTarget:(id)target {
	firstButton.target = secondButton.target = thirdButton.target = target;
}

- (void)setButtonsAction:(SEL)action {
	firstButton.action = secondButton.action = thirdButton.action = action;
}

- (void)setShelveDictionary:(NSDictionary *)dictionary {
	NSNumber *date = [dictionary objectForKey:@"revtime"];
	if (date)
		[self setDate:date];
	else
		dateDetailLabel.stringValue = @"...";
}

- (void)setVersionDictionary:(NSDictionary *)dictionary {
	
	[self setVersionNumber:[dictionary objectForKey:@"rev"]];
	
	NSString *message = [dictionary objectForKey:@"desc"] ?: @"";
	titleLabel.toolTip = message;
	titleLabel.stringValue = message;
		
	authorDetailTitle.stringValue = @"MODIFIED BY";
	authorDetailLabel.stringValue = [dictionary objectForKey:@"user"] ?: @"";

	[self setDate:[dictionary objectForKey:@"time"]];
	
	NSString *action = [dictionary objectForKey:@"headAction"];
	actionDetailTitle.stringValue = @"ACTION";
	actionDetailLabel.stringValue = action ?: @"";
	
	BOOL editAction = [action isEqualToString:@"edit"];
	CGRect lastFrame = editAction ? dateDetailLabel.frame : actionDetailLabel.frame;
	[actionDetailTitle setHidden:editAction];
	[actionDetailLabel setHidden:editAction];
	
	CGRect frame;
	frame = tagsTitle.frame;
	frame.origin.y = lastFrame.origin.y - frame.size.height - 9.0f;
	tagsTitle.frame = frame;
	
	frame = tagsView.frame;
	frame.origin.y = tagsTitle.frame.origin.y - frame.size.height;
	tagsView.frame = frame;
	
	rowHeight = self.frame.size.height - CGRectGetMaxY(tagsTitle.frame);
}

- (void)setTags:(NSArray *)tags {
	
	//	BOOL hidden = !tags.count;
	//	[tagsTitle setHidden:hidden];
	//	[tagsView setHidden:hidden];
	
	if (tags) {
		[tagsView setTags:tags];
		rowHeight = self.frame.size.height - tagsView.frame.origin.y;
	}
}

- (void)setStatusColor:(NSColor *)color {
	statusColor = color;
	[leftBox setFillColor:statusColor];
}

@end
