package com.perforce.api;

import java.io.*;
import java.util.*;
import java.text.*;

/*
 * Copyright (c) 2001, Perforce Software, All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * Representation of a source control file.
 *
 * @see Hashtable
 * @author <a href="mailto:david@markley.cc">David Markley</a>
 * @version $Date: 2002/07/11 $ $Revision: #6 $
 */
public final class FileEntry extends SourceControlObject {
  private String depot_path = null;
  private String client_path = null;
  private String description = "";
  private String owner = "";
  private FileEntry source = null;
  private int head_change = -1;
  private int head_rev = 0;
  private String head_type = "unknown";
  private long head_time = 0;
  private int have_rev = 0;
  private int other_cnt = 0;
  private String head_action = "";
  private String open_action = "";
  private Vector others;
  private static HashDecay fentries;
  private String file_content = "";
  private DateFormat fmt =
    DateFormat.getDateTimeInstance(DateFormat.SHORT,
                                   DateFormat.MEDIUM);
   private int change;


  /** Default no-argument constructor. */
  public FileEntry() {
    this((Env) null);
  }

  /**
   * Constructs a file entry using the environment.
   *
   * @param env Source control environement to use.
   */
  public FileEntry(Env env) {
    super(env);
    if (null == others) {
      others = new Vector();
    }
  }

  /**
   * Constructs a file entry using the environment and path.
   *
   * @param env Source control environement to use.
   * @param p Path to the file.
   */
  public FileEntry(Env env, String p) {
    this(env);
    if (p.startsWith("//")) {
      depot_path = p;
    }
    else {
      client_path = p;
    }
  }

  /**
   * Constructs a file entry using the path.
   *
   * @param p Path to the file.
   */
  public FileEntry(String p) {
    this(null, p);
  }

  private static HashDecay setCache() {
    if (null == fentries) {
      fentries = new HashDecay(120000);
      fentries.start();
    }
    return fentries;
  }

  public HashDecay getCache() {
    return setCache();
  }

  /** @param d the decription for this file */
  public void setDescription(String d) {
    description = d;
  }

  /** @return the decription for this file */
  public String getDescription() {
    return description;
  }

  /** @param o the owner for this file */
  public void setOwner(String o) {
    int pos = o.indexOf('@');
    owner = o;
    if (-1 != pos) {
      owner = owner.substring(0, pos);
    }
  }

  /** @return the owner for this file */
  public String getOwner() {
    return owner;
  }

  /** @param fent the source file entry associated with this file. */
  public void setSource(FileEntry fent) {
    source = fent;
  }

  /** @return the source file entry associated with this file. */
  public FileEntry getSource() {
    return source;
  }

  /** @param type the head revision type for this file. */
  public void setHeadType(String type) {
    this.head_type = type;
  }

  /** @return the head revision type for this file. */
  public String getHeadType() {
    return this.head_type;
  }

  /**
   * @param date the head date for this file. The expected format for the date
   * is yyyy/MM/dd. The time will default to 12:00:00 AM.
   */
  public void setHeadDate(String date) {
    SimpleDateFormat formatter = new SimpleDateFormat ("yyyy/MM/dd");
    // Parse the previous string back into a Date.
    Date hDate = formatter.parse(date, new ParsePosition(0));
    this.head_time = hDate.getTime() / 1000;
  }

  /**
   * @return a String representation of date for the head revsision of the
   * file. The format is yyyy/MM/dd.
   */
  public String getHeadDate() {
    SimpleDateFormat formatter = new SimpleDateFormat ("yyyy/MM/dd");
    return formatter.format(new Date(this.head_time * 1000));
  }

  /** @param time the head revision time for this file. */
  public void setHeadTime(long time) {
    this.head_time = time;
  }

  /** @return the head revision time for this file. */
  public long getHeadTime() {
    return this.head_time;
  }

  /**
   * Sets the format used by the getHeadTimeString method. The format of
   * this string is that of the SimpleDateFormat class.
   * <p>
   * An example format would be setTimeFormat("MM/dd HH:mm:ss");
   *
   * @see SimpleDateFormat
   */
  public void setTimeFormat(String format) {
    if (null == format) return;
    fmt = new SimpleDateFormat(format);
  }

  /** @return the head revision time as a <code>String</code> for this file. */
  public String getHeadTimeString() {
    Date d = new Date(this.head_time * 1000);
    return fmt.format(d);
  }

  /** @param action the head revision action for this file. */
  public void setHeadAction(String action) {
    this.head_action = action;
  }

  /** @return the head revision action for this file. */
  public String getHeadAction() {
    return this.head_action;
  }

  /** @param change the head revision change number for this file. */
  public void setHeadChange(int change) {
    this.head_change = change;
  }

  /** @return the head revision change number for this file. */
  public int getHeadChange() {
    return this.head_change;
  }

  /** @param rev the head revision number for this file. */
  public void setHeadRev(int rev) {
    this.head_rev = rev;
  }

  /** @return the head revision number for this file. */
  public int getHeadRev() {
    return this.head_rev;
  }

  /** @param rev the revision number the client has for this file. */
  public void setHaveRev(int rev) {
    this.have_rev = rev;
  }

  /** @return the revision number the client has for this file. */
  public int getHaveRev() {
    return this.have_rev;
  }

  /** @param action the action the client used to open this file. */
  public void setAction(String action) {
    this.open_action = action;
  }

  /** @return the action the client used to open this file. */
  public String getAction() {
    return this.open_action;
  }

  /**
   * Sets the depot path for this file.
   *
   * @param p  path for this file in the depot.
   */
  public void setDepotPath(String p) {
    this.depot_path = p;
  }

  /** @return the depot path for this file. */
  public String getDepotPath() {
    return depot_path;
  }

  /** @param path the path in local format. Uses the local path delimeter. */
  public static String localizePath(String path) {
    return path.replace('/', File.separatorChar);
  }

  /** @return the path in depot format. Uses the depot delimeter: '/'. */
  public static String depotizePath(String path) {
    return path.replace(File.separatorChar, '/');
  }

  /**
   * Resolves this file. If the force flag is false, an auto-resolve is
   * attempted (p4 resolve -am). If the force flag is true, an "accept theirs"
   * resolve is completed (p4 resolve -at).
   *
   * @param force Indicates whether the resolve should be forced.
   * @return a <code>String</code> value
   * @exception IOException if an error occurs
   */
  public String resolve(boolean force) throws IOException {
    StringBuffer sb = new StringBuffer();
    String[] rescmd = new String[4];
    rescmd[0] = "p4";
    rescmd[1] = "resolve";
    rescmd[2] = (force
                 || -1 != (getHeadType().indexOf("binary"))
                 || -1 != (getHeadType().indexOf("link"))
                 ) ? "-at" : "-am";
    rescmd[3] = getDepotPath();
    P4Process p = new P4Process(getEnv());
    p.exec(rescmd);
    for (String line = p.readLine(); line != null; line = p.readLine()) {
      sb.append(line);
      sb.append('\n');
    }
    p.close();
    return sb.toString();
  }

  /**
   * Forces a resolve on a set of files. The <code>Enumeration</code> contains
   * the set of <code>FileEntry</code> objects that need resolved.
   *
   * @param env Source control environment to use.
   * @param en <code>Enumeration</code> of <code>FileEntry</code>.
   * @return a <code>String</code> value
   * @exception IOException if an error occurs
   */
  public static String resolveAT(Env env, Enumeration en) throws IOException {
    StringBuffer sb = new StringBuffer();
    String[] rescmd = { "p4", "-x", "-", "resolve", "-at" };
    P4Process p = new P4Process(env);
    p.exec(rescmd);
    while (en.hasMoreElements()) {
      final FileEntry fent = (FileEntry) en.nextElement();
      p.println(fent.getDepotPath());
      Debug.notify("resolveAT(): " + fent.getDepotPath());
    }
    p.println("\032\n\032");
    p.flush();
    p.outClose();
    Debug.notify("FileEntry.resolveAT(): Reading more lines.");
    for (String line = p.readLine(); line != null; line = p.readLine()) {
      sb.append(line);
      sb.append('\n');
    }
    p.close();
    return sb.toString();
  }

  /**
   * Resolves all the files in the path. The flags are used by the 'p4 resolve'
   * command to resolve any files in the path. This is just a simple way to
   * execute the 'p4 resolve' command.
   *
   * @param env Source control environment to use.
   * @param flags 'p4 resolve' command flags.
   * @param path  Path over which to resolve. May include wildcards.
   * @return a <code>String</code> value
   * @exception IOException if an error occurs
   */
  public static String resolveAll(Env env, String flags, String path)
    throws IOException
  {
    StringBuffer sb = new StringBuffer();
    String[] rescmd = { "p4", "resolve", flags, path};
    P4Process p = new P4Process(env);
    p.exec(rescmd);
    Debug.notify("FileEntry.resolveAll(): Reading more lines.");
    for (String line = p.readLine(); line != null; line = p.readLine()) {
      sb.append(line);
      sb.append('\n');
    }
    p.close();
    return sb.toString();
  }


  /** @return the file name. */
  public String getName() {
    String path = getDepotPath();
    if (null == path) {
      path = getClientPath();
    }
    if (null == path) {
      return "";
    }
    final int pos = path.lastIndexOf('/');
    if (-1 == pos) {
      return path;
    }
    return path.substring(pos + 1);
  }

  /**
   * Sets the client path for this file.
   *
   * @param p  path for this file on the client system.
   */
  public void setClientPath(String p) {
    this.client_path = p;
  }

  /** @return the client path for this file. */
  public String getClientPath() {
    return client_path;
  }

  /**
   * Gets the file information for the specified path.
   *
   * @param p  Path of the file to gather information about.
   * @return a <code>FileEntry</code> value
   */
  public static synchronized FileEntry getFile(String p) {
    FileEntry f = new FileEntry(p);
    f.sync();
    return f;
  }

  /**
   * Gets the list of files for the path. The path may include wildcards.
   *
   * @param env Source control environment to use.
   * @param path Path for set of files.
   * @return a <code>Vector</code> value
   */
  public static Vector getFiles(Env env, String path) {
    Vector v = null;
    String[] cmd = { "p4", "fstat", "-C", path + "%1" };
    if (null == path) return null;

    try {
      P4Process p = new P4Process(env);
      p.exec(cmd);
      v = parseFstat(null, p, true);
      p.close();
    } catch (IOException ex) { Debug.out(Debug.ERROR, ex); }
    return v;
  }

  /**
   * Gets a list of <code>FileEntry</code> objects that represent the
   * history of the specified file.
   *
   * @param env Source control environment to use.
   * @param path  Path to the file. Must be specific. No wildcards.
   * @return a <code>Vector</code> value
   */
  public static Vector getFileLog(Env env, String path) {
    String[] cmd = { "p4", "filelog", path };
    FileEntry fent = null, tmpent = null;
    Vector v = new Vector();
    int beg, end;

    if (null == path) return v;
    try {
      P4Process p = new P4Process(env);
      p.setRawMode(true);
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        line = line.trim();
        if (line.startsWith("info2: ") && fent != null) {
          tmpent = new FileEntry(env);
          beg = 8;
          end = line.indexOf(' ', beg);
          if (-1 == end) {
            continue;
          }
          tmpent.setHeadAction(line.substring(beg, end));
          beg = end;
          end = line.indexOf("from ");
          if (-1 == end) {
            tmpent.setDepotPath(path);
          }
          else {
            beg = end + 5;
            end = line.indexOf('#', beg);
            if (-1 == end) {
              tmpent.setDepotPath(line.substring(beg));
            }
            else {
              tmpent.setDepotPath(line.substring(beg, end));
            }
          }
          end = line.lastIndexOf('#');
          if (-1 != end) {
            beg = line.lastIndexOf('#', end - 1);
            if (-1 != beg) {
              tmpent.setHaveRev(Integer.parseInt(line.substring(beg + 1,
                                                                end - 1)));
            }
            tmpent.setHeadRev(Integer.parseInt(line.substring(end + 1)));
          }
          fent.setSource(tmpent);
        }
        else if (line.startsWith("info1: ")) {
          if (fent != null) {
            v.addElement(fent);
          }
          fent = new FileEntry(env);
          fent.setDepotPath(path);
          StringTokenizer st = new StringTokenizer(line.substring(8));
          fent.setHeadRev(Integer.parseInt(st.nextToken()));
          st.nextToken(); // change
          fent.setHeadChange(Integer.parseInt(st.nextToken()));
          fent.setHeadAction(st.nextToken());
          st.nextToken(); // on
          fent.setHeadDate(st.nextToken());
          st.nextToken(); // by
          fent.setOwner(st.nextToken());
          String tmp = st.nextToken();
          fent.setHeadType(tmp.substring(1, tmp.length() - 1));
          if (1 < (end = line.lastIndexOf('\''))) {
            if (-1 < (beg = line.lastIndexOf('\'', end - 1))) {
              if (end - beg - 1 > 0) {
                fent.setDescription(line.substring(beg + 1, end - 1));
              }
            }
          }
        }
      }
      p.close();
    } catch (IOException ex) { Debug.out(Debug.ERROR, ex); }
    if (fent != null && fent.getDepotPath() != null) {
      v.addElement(fent);
    }
    return v;
  }

  /**
   * Opens the file on the path for edit under the change. If the change is
   * null, the file is opened under the default changelist.
   *
   * @param env  P4 Environment
   * @param path Depot or client path to the file being opened for edit.
   * @param sync If true, the file will be sync'd before opened for edit.
   * @param force If true, the file will be opened for edit even if it
   *    isn't the most recent version.
   * @param lock If true, the file will be locked once opened.
   * @param chng The change that the file will be opened for edit in.
   * @return a <code>FileEntry</code> value
   * @exception IOException if an error occurs
   */
  public static FileEntry openForEdit(Env env, String path, boolean sync, boolean force, boolean lock, Change chng)
      throws IOException
  {
    if (sync) {
      FileEntry.syncWorkspace(env, path);
    }
    FileEntry fent = new FileEntry(env, path);
    fent.openForEdit(force, lock, chng);
    return fent;
  }

  /**
   * Opens this file for edit.
   *
   * @exception IOException if an error occurs
   * @see #openForEdit(boolean, boolean, Change)
   */
  public void openForEdit() throws IOException {
    openForEdit(true, false, null);
  }

  /**
   * Opens this file for edit.
   *
   * @param force If true, the file will be opened for edit even if it
   *    isn't the most recent version.
   * @param lock If true, the file will be locked once opened.
   * @exception IOException if an error occurs
   * @see #openForEdit(boolean, boolean, Change)
   */
  public void openForEdit(boolean force, boolean lock) throws IOException {
    openForEdit(force, lock, null);
  }

  /**
   * Opens this file for edit.
   * @param force If true, the file will be opened for edit even if it
   *    isn't the most recent version.
   * @param lock If true, the file will be locked once opened.
   * @param chng a <code>Change</code> value
   * @exception IOException if an error occurs
   */
  public void openForEdit(boolean force, boolean lock, Change chng)
   throws IOException
  {
    String[] cmd1;
    String[] cmd2;
      P4Process p;
    int i = 0;
    sync();
    if (force) {
      cmd1 = new String[4];
      cmd1[2] = "-f";
    } else {
      cmd1 = new String[3];
    }
    cmd1[0] = "p4";
    cmd1[1] = "sync";
    cmd1[cmd1.length - 1] = getDepotPath();

    cmd2 = new String[(null == chng) ? 3 : 5];
    cmd2[i++] = "p4";
    cmd2[i++] = "edit";
    if (chng != null) {
      cmd2[i++] = "-c";
      cmd2[i++] = String.valueOf(chng.getNumber());
    }
    cmd2[i++] = getClientPath();

    p = new P4Process(getEnv());
    p.exec(cmd1);
    for (String line = p.readLine(); line != null; line = p.readLine()) {}
    p.close();
    p = new P4Process(getEnv());
    p.exec(cmd2);
    for (String line = p.readLine(); line != null; line = p.readLine()) {}
    p.close();
    if (lock) obtainLock();
  }

  /**
   * Opens this file for delete.
   *
   * @exception IOException if an error occurs
   * @see #openForDelete(boolean, boolean, Change)
   */
  public void openForDelete() throws IOException {
    openForDelete(true, false, null);
  }

  /**
   * Opens this file for delete.
   *
   * @param force If true, the file will be opened for edit even if it
   *    isn't the most recent version.
   * @param lock If true, the file will be locked once opened.
   * @exception IOException if an error occurs
   * @see #openForDelete(boolean, boolean, Change)
   */
  public void openForDelete(boolean force, boolean lock) throws IOException {
    openForDelete(force, lock, null);
  }

  /**
   * Opens this file for delete.
   * @param force If true, the file will be opened for edit even if it
   *    isn't the most recent version.
   * @param lock If true, the file will be locked once opened.
   * @param chng a <code>Change</code> value
   * @exception IOException if an error occurs
   */
  public void openForDelete(boolean force, boolean lock, Change chng)
    throws IOException
  {
    String[] cmd1;
    String[] cmd2;
    P4Process p;
    int i = 0;
    sync();
    if (force) {
      cmd1 = new String[4];
      cmd1[2] = "-f";
    } else {
      cmd1 = new String[3];
    }
    cmd1[0] = "p4";
    cmd1[1] = "sync";
    cmd1[cmd1.length - 1] = getDepotPath();

    cmd2 = new String[(null == chng) ? 3 : 5];
    cmd2[i++] = "p4";
    cmd2[i++] = "delete";
    if (chng != null) {
      cmd2[i++] = "-c";
      cmd2[i++] = String.valueOf(chng.getNumber());
    }
    cmd2[i++] = getClientPath();

    p = new P4Process(getEnv());
    p.exec(cmd1);
    for (String line = p.readLine(); line != null; line = p.readLine()) {}
    p.close();
    p = new P4Process(getEnv());
    p.exec(cmd2);
    for (String line = p.readLine(); line != null; line = p.readLine()) {}
    p.close();
    if (lock) obtainLock();
  }

  /**
   * Obtains the lock for this file. The file must have been opened for
   * edit prior to this method being called.
   * @exception IOException if an error occurs
   */
  public void obtainLock() throws IOException {
    String[] cmd = { "p4", "lock", getDepotPath() };
    P4Process p = new P4Process(getEnv());
    p.exec(cmd);
    for (String line = p.readLine(); line != null; line = p.readLine()) {
    }
    p.close();
  }

  /**
   * Opens the file on the path for add under the change. If the change is
   * null, the file is opened under the default changelist.
   *
   * @param env  P4 Environment
   * @param path Depot or client path to the file being opened for add.
   * @param chng The change that the file will be opened for add in.
   * @return a <code>FileEntry</code> value
   * @exception IOException if an error occurs
   */
  public static FileEntry openForAdd(Env env, String path, Change chng)
    throws IOException
  {
    FileEntry fent = new FileEntry(env, path);
    fent.openForAdd(chng);
    return fent;
  }

  /**
   * Opens this file for addition.
   *
   * @exception IOException if an error occurs
   * @see #openForAdd(Env, String, Change)
   */
  public void openForAdd() throws IOException {
    openForAdd(null);
  }

  /**
   * Opens this file for addition.
   *
   * @param chng a <code>Change</code> value
   * @exception IOException if an error occurs
   * @see #openForAdd(Env, String, Change)
   */
  public void openForAdd(Change chng) throws IOException {
    String[] cmd = new String[(null == chng) ? 3 : 5];
    cmd[0] = "p4";
    cmd[1] = "add";
    if (chng != null) {
      cmd[2] = "-c";
      cmd[3] = String.valueOf(chng.getNumber());
    }
    cmd[cmd.length - 1] = getClientPath();

    if (null == getClientPath()) {
      throw new IOException("No Client Path");
    }

    P4Process p = new P4Process(getEnv());
    p.exec(cmd);
    for (String line = p.readLine(); line != null; line = p.readLine()) {
    }
    p.close();
  }

  /**
   * Checks in a file that has already been opened on the client using the
   * description given. A new changelist is created and used for this
   * submission. The returned <code>FileEntry</code> contains the latest
   * information for the checked-in file.
   * @param env an <code>Env</code> value
   * @param path a <code>String</code> value
   * @param description a <code>String</code> value
   * @return a <code>FileEntry</code> value
   * @exception PerforceException if an error occurs
   */
  public static FileEntry checkIn(Env env, String path, String description)
    throws PerforceException
  {
    FileEntry fent = new FileEntry(env, path);
    Change chng = new Change(env);
    chng.setDescription(description);
    chng.addFile(fent);
    chng.submit();
    fent.sync();
    return fent;
  }

  /**
   * Reopens the file with the new type or in the new change list.
   * @param type a <code>String</code> value
   * @param chng a <code>Change</code> value
   * @exception PerforceException if an error occurs
   */
  public void reopen(String type, Change chng) throws PerforceException {
    if (null == getClientPath()) {
      sync();
      if (null == getClientPath()) {
        throw new PerforceException("No Client Path");
      }
    }

    if (null == type && null == chng) return;
    String[] cmd = new String[(null == type || null == chng) ? 5 : 7];
    cmd[0] = "p4";
    cmd[1] = "reopen";
    int i = 2;
    if (type != null) {
      cmd[i++] = "-t";
      cmd[i++] = type;
    }
    if (chng != null) {
      cmd[i++] = "-c";
      cmd[i++] = String.valueOf(chng.getNumber());
    }
    cmd[i++] = getClientPath();

    P4Process p = new P4Process(getEnv());
    try {
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        if ( (-1 != line.indexOf("not opened on this client")) ||
             (-1 != line.indexOf("Invalid file type")) ||
             (-1 != line.indexOf("unknown")) ) {
          throw new PerforceException(line);
        }
      }
    } catch (IOException ex) {
      throw new PerforceException(ex.getMessage());
    } finally {
      if (p != null) {
        try { p.close(); } catch (IOException ioex) { /* Ignored Exception */ }
      }
    }
  }


  /**
   * Reverts this file.
   * @return <code>true</code> for success, <code>false</code> for failure
   */
  public boolean revert() {
    String[] cmd1 = { "p4", "revert", getDepotPath() };
    String[] cmd2 = { "p4", "sync", getDepotPath() + "#none" };
    try {
      P4Process p = new P4Process(getEnv());
      p.exec(cmd1);
      for (String line = p.readLine(); line != null; line = p.readLine()) {}
      p.close();
      p = new P4Process(getEnv());
      p.exec(cmd2);
      for (String line = p.readLine(); line != null; line = p.readLine()) {}
      p.close();
    } catch (IOException ex) {
      Debug.out(Debug.ERROR, ex);
      return false;
    }
    return true;
  }

  /**
   * @return a list of files that are open for edit or add. The list is
   * a <code>Vectore</code> of <code>FileEntry</code> objects.
   * The only information that is valid for the object will be the path,
   * until the {@link #sync() sync} method is called.
   */
  public static Vector getOpened() {
    return getOpened(null, true, false, -1, null);
  }

  /**
   * @return a list of files that are open for edit or add. The list is
   * a <code>Vectore</code> of <code>FileEntry</code> objects.
   *<p>
   * Getting the stats for each <code>FileEntry</code> is a more expensive
   * operation. By default, this is not done. What this means is that the
   * only information that is valid for the object will be the path, until the
   * {@link #sync() sync} method is called.
   *
   * @param env  Source control environment to use.
   * @param stat  Indicates that file statistics should be gathered.
   */
  public static Vector getOpened(Env env, boolean stat) {
    return getOpened(env, stat, false, -1, null);
  }

  /**
   * @return a list of files that are open for edit or add. The list is
   * a <code>Vector</code> of <code>FileEntry</code> objects.
   *<p>
   * Getting the stats for each <code>FileEntry</code> is a more expensive
   * operation. By default, this is not done. What this means is that the
   * only information that is valid for the object will be the path, until the
   * {@link #sync() sync} method is called.
   *<p>
   * If changelist is 0, all the changes in the default changelist are
   * returned. If it is less than 0, all opened files are returned.
   *
   * @param env  Source control environment to use.
   * @param stat  Indicates that file statistics should be gathered.
   * @param all  Indicates that all open files should be returned.
   * @param changelist  If non-zero, show files open in this changelist.
   * @param files If non-null, show files open in this
   * <code>Vector</code> of <code>FileEntry</code> objects.
   */
  public static Vector getOpened(Env env, boolean stat, boolean all,
                                 int changelist, Vector files) {
    Vector v = new Vector();
    int i = 0, cnt = 2;
    if (all) cnt++;
    if (0 <= changelist) cnt += 2;
    if (files != null) cnt += files.size();
    String[] cmd = new String[cnt];
    cmd[i++] = "p4";
    cmd[i++] = "opened";
    if (all) cmd[i++] = "-a";
    if (0 <= changelist) {
      cmd[i++] = "-c";
      cmd[i++] = (0 == changelist) ? "default" : String.valueOf(changelist);
    }
    if (files != null) {
      Enumeration en = files.elements();
      while (en.hasMoreElements()) {
        cmd[i++] = (String) en.nextElement();
      }
    }
    try {
      P4Process p = new P4Process(env);
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        if (!line.startsWith("//")) continue;
        StringTokenizer st = new StringTokenizer(line, "#");

        String str = st.nextToken();
        if (null == str) continue;
        FileEntry fent = new FileEntry(env, str);

	str = st.nextToken("# \t");
        if (null == str) continue;
        fent.setHeadRev(Integer.valueOf(str).intValue());
        st.nextToken(" \t"); // Should be the dash here.
	str = st.nextToken();
        if (null == str)  continue;
        fent.setHeadAction(str);
	str = st.nextToken();
        if (null == str)  continue;
        if (str.equals("default")) {
          fent.setHeadChange(-1);
          st.nextToken(); // Change here.
        }
        else if (str.equals("change")) {
	  str = st.nextToken();
          if (null == str)  continue;  // Change number
          fent.setHeadChange(Integer.valueOf(str).intValue());
        }
	str = st.nextToken(" \t()");
        if (null == str)  continue;
        fent.setHeadType(str);
        // Insertion sort...slow but effective.
        for (i = 0; i < v.size(); i++) {
          if (((FileEntry) v.elementAt(i)).getHeadChange() > fent.getHeadChange()) break;
        }
        v.insertElementAt(fent, i);
      }
      p.close();
    } catch (IOException ex) { Debug.out(Debug.ERROR, ex); }
    if (stat) {
      Enumeration en = v.elements();
      while (en.hasMoreElements()) {
        final FileEntry fent = (FileEntry) en.nextElement();
        fent.setEnv(env);
        fent.sync();
      }
    }
    return v;
  }

  /**
   * No-op. This makes no sense for a FileEntry.
   */
  public void commit() {
  }

  /**
   * @deprecated
   * @see #syncWorkspace(Env, String)
   */
  public String syncMySpace(Env env, String path) throws IOException {
    return FileEntry.syncWorkspace(env, path);
  }

  /**
   * Returns a <code>Vector</code> of <code>FileEntry</code> objects that
   * reflect what files were changed by the sync process. If path is null,
   * the entire workspace is synchronized to the head revision. The path may
   * contain wildcard characters, as with the command line 'p4 sync' command.
   *
   * @param env  Source control environment.
   * @param path Path to synchronize. May include wildcards.
   * @return a <code>Vector</code> value
   * @exception IOException if an error occurs
   */
  public static Vector synchronizeWorkspace(Env env, String path) throws IOException {
    String [] cmd;
    if (null == path || path.trim().equals("")) {
      cmd = new String[2];
    } else {
      cmd = new String[3];
      cmd[2] = path;
    }
    cmd[0] = "p4";
    cmd[1] = "sync";

    int pos1, pos2;
    Vector v = new Vector();
    try {
      P4Process p = new P4Process(env);
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        if (!line.startsWith("//")) continue;
        pos1 = 0;
	pos2 = line.indexOf('#');
        if (-1 == pos2) continue;
        final FileEntry fent = new FileEntry(env, line.substring(pos1, pos2));
        pos1 = pos2 + 1;
	pos2 = line.indexOf(' ', pos1);
        if (-1 == pos2) continue;
        try {
          fent.setHeadRev(Integer.parseInt(line.substring(pos1, pos2)));
        } catch (NumberFormatException ex) {
          continue;
        }
        pos1 = pos2 + 1;
        if (-1 != (pos2 = line.indexOf("updating ")) ||
            -1 != (pos2 = line.indexOf("added as "))) {
          fent.setClientPath(line.substring(pos2 + 9).trim());
        }
        if (fent != null) {
          v.addElement(fent);
        }
      }
      p.close();
    } catch (IOException ex) {
      Debug.out(Debug.ERROR, ex);
      throw ex;
    }
    return v;
  }

  /**
   * Synchronizes the workspace.
   *
   * @param env  Source control environment.
   * @param path Path to synchronize. May include wildcards.
   * @return a <code>String</code> value
   * @exception IOException if an error occurs
   */
  public static String syncWorkspace(Env env, String path) throws IOException {
    String [] cmd;
    if (null == path || path.trim().equals("")) {
      cmd = new String[2];
    } else {
      cmd = new String[3];
      cmd[2] = path;
    }
    cmd[0] = "p4";
    cmd[1] = "sync";

    StringBuffer buf = new StringBuffer();
    try {
      P4Process p = new P4Process(env);
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        buf.append(line).append('\n');
      }
      p.close();
    } catch (IOException ex) {
      Debug.out(Debug.ERROR, ex);
      throw ex;
    }
    return buf.toString();
  }

  /**
   * @return a <code>String</code> that contains this file's contents. This
   * only works well for text files.
   */
  public String getFileContents() {
    return getFileContents(getEnv(), getDepotPath());
  }

  /**
   * Gets a <code>String</code> that contains this file's contents. This
   * only works well for text files.
   *
   * @param env  Source control environment.
   * @param path Path to the file. Must be specific. No wildcards.
   * @return a <code>String</code> value
   */
  public String getFileContents(Env env, String path) {
    StringBuffer buf = new StringBuffer();
    String[] cmd = { "p4", "print", path };
    try {
      P4Process p = new P4Process(env);
      p.setRawMode(true);
      p.exec(cmd);
      for (String line = p.readLine(); line != null; line = p.readLine()) {
        if (line.startsWith("text: ")) {
          buf.append(line.substring(6));
          if (!line.endsWith("\n")) buf.append('\n');
        }
      }

      if (0 != p.close()) {
        throw new IOException("P4 exited with and error:" + p.getExitCode());
      }
    } catch (IOException ex) {
      Debug.out(Debug.ERROR, ex);
    }
    file_content = buf.toString();
    return file_content;
  }

  public void sync() {
    String l;
    String[] cmd = { "p4", "fstat", "path" };
    if (depot_path != null) {
      cmd[2] = depot_path;
    }
    else if (client_path != null) {
      cmd[2] = client_path;
    } else {
      return;
    }
    if (0 != head_rev) {
      cmd[2] += "#" + head_rev;
    }
    try {
      P4Process p = new P4Process(getEnv());
      p.exec(cmd);
      parseFstat(this, p, false);
      if (0 != p.close()) {
        throw new IOException("P4 exited with an error:" + p.getExitCode());
      }
      this.inSync();
    } catch (IOException ex) { Debug.out(Debug.ERROR, ex); }
  }

  /**
   * Useful method for parsing that lovely fstat format information.
   */
  private static Vector parseFstat(FileEntry fe, P4Process p, boolean igndel) {
    FileEntry nfe = fe;
    Vector v = new Vector();
    String dataname, datavalue;
    boolean multiple = false;

    if (null == p) return null;
    if (null == nfe) nfe = new FileEntry(p.getEnv());

    for (String line = p.readLine(); line != null; line = p.readLine()) {
      StringTokenizer st = new StringTokenizer(line, " ");

      if (!st.hasMoreTokens()) continue;
      dataname =  st.nextToken();
      if (!st.hasMoreTokens()) continue;
      datavalue = st.nextToken();

      switch (dataname.charAt(0)) {
      case 'c':
        if (dataname.equals("change")) {
          nfe.change = datavalue.equals("default") ? 0 :
            Integer.parseInt(datavalue);
        }
        else if (dataname.equals("clientFile")) {
          nfe.setClientPath(datavalue);
        }
        continue;

      case 'd':
        if (dataname.equals("depotFile")) {
          if (multiple) nfe = new FileEntry(p.getEnv());
          nfe.setDepotPath(datavalue);
          v.add(nfe);
          multiple = true;
        }
        continue;

      case 'h':
        if (dataname.equals("headAction")) {
          nfe.setHeadAction(datavalue);
        }
        else if (dataname.equals("headChange")) {
          nfe.setHeadChange(Integer.parseInt(datavalue));
        }
        else if (dataname.equals("headRev")) {
          nfe.setHeadRev(Integer.parseInt(datavalue));
        }
        else if (dataname.equals("headType")) {
          nfe.setHeadType(datavalue);
        }
        else if (dataname.equals("headTime")) {
          nfe.setHeadTime(Long.parseLong(datavalue));
        }
        else if (dataname.equals("haveRev")) {
          nfe.setHaveRev(Integer.parseInt(datavalue));
        }
        continue;

      case 'o':
        if (dataname.equals("unresolved")) {
        } else if (dataname.equals("otherOpen")) {
        } else if (dataname.equals("otherLock")) {
        } else if (dataname.equals("ourLock")) {
        }
        continue;

      default:
        if (dataname.equals("action")) {
          nfe.setAction(datavalue);
         continue;
        }
        else if (dataname.equals("change")) {
        continue;
        }
      Debug.warn(line);
      }
    }
    return v;
  }

  public String toString() {
    return depot_path + '\n' + client_path + "\nothers: " + other_cnt;
  }

  public String toXML() {
    StringBuffer sb = new StringBuffer(512).append("<file>");
    sb.append("<have rev=\"").append(getHaveRev())
      .append("\" action=\"").append(getAction())
      .append("\"/>");
    sb.append("<head rev=\"").append(getHeadRev())
      .append("\" change=\"").append(getHeadChange())
      .append("\" type=\"").append(getHeadType())
      .append("\" action=\"").append(getHeadAction())
      .append("\" time=\"").append(getHeadTimeString())
      .append("\"/>");
    sb.append("<path type=\"depot\">")
      .append(getDepotPath())
      .append("</path>");
    sb.append("<path type=\"client\">")
      .append(getClientPath())
      .append("</path>");
    if (getDescription() != null) {
      sb.append("<description>")
        .append(getDescription())
        .append("</description>");
    }
    return sb.append("</file>").toString();
  }

   public int getChange() {
      return change;
   }
}
