#!/bin/bash
set -u

declare ThisScript="${0##*/}"
declare Version=1.0.1
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare StandbyHost=bos-helix-02
declare -i StandbyLogCount=0
declare -a StandbyLogList
declare EdgeHost=syd-helix-04
declare -i EdgeLogCount=0
declare -a EdgeLogList

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript v$Version]: Call p4verify.sh on remote host."

StandbyLogList[StandbyLogCount]="/p4/1/logs/p4verify.log"
StandbyLogCount+=1
EdgeLogList[EdgeLogCount]="/p4/1/logs/p4verify.log"
EdgeLogCount+=1

msg "Running: 'p4verify.sh 1 -recent=head' on standby host $StandbyHost."
ssh -q "$StandbyHost" /p4/common/bin/p4verify.sh 1 -recent=head ||\
   errmsg "Remote call to p4verify.sh on host $StandbyHost indicated an error."

msg "Running: 'p4verify.sh 1 -recent=head' on edge host $EdgeHost."
ssh -q "$EdgeHost" /p4/common/bin/p4verify.sh 1 -recent=head ||\
   errmsg "Remote call to p4verify.sh on host $EdgeHost indicated an error."

msg "Checking for syntax and unbound errors in scripts used in this test on standby host."
for log in "${StandbyLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $StandbyHost:$log"
   ssh -q "$StandbyHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $StandbyHost:$log"
   ssh -q "$StandbyHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
   ssh -q "$StandbyHost" "grep 'line: ' $log" && \
      errmsg "Found 'line:' error in $log"
done

msg "Checking for syntax and unbound errors in scripts used in this test on edge host."
for log in "${EdgeLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
   ssh -q "$EdgeHost" "grep 'line: ' $log" && \
      errmsg "Found 'line:' error in $log"
done

msg "\\nTest script $ThisScript v$Version reported $ErrorCount errors."

exit "$ErrorCount"
