# -*- encoding: UTF8 -*-
# Test harness for CheckChangeDesc.py

from __future__ import print_function

import sys
import unittest
import os
import re

import P4
from p4testutils import TestCase, P4Server, localDirectory, create_file, append_to_file

parent_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, parent_dir)
from CheckChangeDesc import CheckChangeDesc

os.environ["LOGS"] = "."
LOGGER_NAME = "TestCheckChangeDesc"
LOG_FILE = "log-CheckChangeDesc.log"

python3 = sys.version_info[0] >= 3

class TestCheckChangeDesc(TestCase):
    def __init__(self, methodName='runTest'):
        super(TestCheckChangeDesc, self).__init__(LOGGER_NAME, LOG_FILE, methodName=methodName)

    def setUp(self):
        pass

    def tearDown(self):
        pass


    def setupServer(self):
        self.server = P4Server()
        trig_path = os.path.join(parent_dir, "CheckChangeDesc.py")
        p4 = self.server.p4
        p4.logger = self.logger
        # This works if no spaces in server root pathname!
        port = p4.port.replace('"', '')
        self.logger.debug("port: |%s|" % port)
        triggers = p4.fetch_triggers()
        triggers['Triggers'] = ['check-change-desc change-submit //... "python {}'
                                ' -p %quote%{}%quote% -u {} %change%"'.format(
            trig_path, port, p4.user)]
        self.logger.debug(triggers)
        p4.save_triggers(triggers)
        # Reconnect to pick up changes
        p4.disconnect()
        p4.connect()
        return p4

    def testCheckChangeDesc(self):
        """trigger fires and sends expected info to Swarm"""
        p4 = self.setupServer()

        inside = localDirectory(self.server.client_root, "inside")
        inside_file1 = os.path.join(inside, "inside_file1")
        create_file(inside_file1, 'Test content')

        # Submit without a config file still works fine
        p4.run('add', inside_file1)
        result = p4.run('submit', '-d', 'OS-1234 my change')
        self.assertEqual("1", result[-1]['submittedChange'])

        p4.run('edit', inside_file1)
        append_to_file(inside_file1, "more text")

        invalid_descriptions = [
            "some change",
            "another change\nsecond line",
            "DF-1234",
            "OS-",
            "OS-12", # needs some description
            "OS-12ab",
        ]
        for desc in invalid_descriptions:
            self.invalidDesc(p4, desc)

        valid_descriptions = [
            "OS-12343 my change",
            "OS-123 some text\nsecond line\n# a comment line",
            "OS-1 a",
            "WF-12343 my change",
            "WF-123 some text\nsecond line\n# a comment line",
            "WF-1 ab",
            "OS-15184 - Add new formula to replace the old one in Detail G section. ",
        ]
        for desc in valid_descriptions:
            self.validDesc(p4, inside_file1, desc)

    def validDesc(self, p4, fname, desc):
        try:
            p4.run('revert', fname)
        except:
            pass
        p4.run('edit', fname)
        append_to_file(fname, "more text")
        p4.run('submit', '-d', desc)

    def invalidDesc(self, p4, desc):
        try:
            p4.run('reopen', '-c', 'default', '//...')
            p4.run('submit', '-d', desc)
            self.assertTrue(False, "Expected exception not found: %s" % desc)
        except P4.P4Exception as e:
            self.assertRegex(str(e), r"First line of changelist description")


    def get_changeid(self, result):
        m = re.search("Change ([^ ]+) created", result[0])
        self.assertTrue(m)
        change = m.group(1)
        return change

if __name__ == '__main__':
    unittest.main()
