# -*- encoding: UTF8 -*-
# Test harness for CheckStreamNameFormat.py

from __future__ import print_function

import sys
import unittest
import os
import re

import P4
from p4testutils import TestCase, P4Server, localDirectory, create_file, append_to_file

python3 = sys.version_info[0] >= 3

parent_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, parent_dir)
from CheckStreamNameFormat import CheckStreamNameFormat

os.environ["LOGS"] = "."
LOGGER_NAME = "TestCheckStreamNameFormat"
LOG_FILE = "log-TestCheckStreamNameFormat.log"


class TestCheckStreamNameFormat(TestCase):
    def __init__(self, methodName='runTest'):
        super(TestCheckStreamNameFormat, self).__init__(LOGGER_NAME, LOG_FILE, methodName=methodName)

    def setUp(self):
        self.server = P4Server()
        trigpath = os.path.join(parent_dir, "CheckStreamNameFormat.py")
        self.config_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "~test_config.yaml")
        p4 = self.server.p4
        self.p4 = p4
        p4.logger = self.logger
        depot = p4.fetch_depot("streams")
        depot["Type"] = "stream"
        p4.save_depot(depot)
        # This works if no spaces in server root pathname!
        port = p4.port.replace('"', '')
        self.logger.debug("port: |%s|" % port)
        triggers = p4.fetch_triggers()
        triggers['Triggers'] = ['check-stream-name form-save stream " python {} -p %quote%{}%quote% '
                                '-u {} -c {} %formfile% "'.format(trigpath, port, p4.user, self.config_path),
                                ]
        self.logger.debug(triggers)
        p4.save_triggers(triggers)
        # Reconnect to pick up changes
        p4.disconnect()
        p4.connect()

    def tearDown(self):
        pass

    def testCheckStreamNameFormat(self):
        """check that it works when called as a trigger"""

        p4 = self.p4

        with open(self.config_path, "w") as f:
            f.write("""
msg_invalid_stream_name:
  - ""
  - "Stream name doesn't conform to valid regex format."
  - "See below for list of valid formats"

valid_stream_name_formats:
  - "must_be_this"
""")

        stream = p4.fetch_stream("-tmainline", "//streams/main")
        try:
            p4.save_stream(stream)
            self.assertTrue(False, "Expected exception not found")
        except P4.P4Exception as e:
            self.assertRegex(str(e), r"Stream name doesn't conform to valid regex format")

        with open(self.config_path, "w") as f:
            f.write("""
msg_invalid_stream_name:
  - ""
  - "Stream name doesn't conform to valid regex format."
  - "See below for list of valid formats"

valid_stream_name_formats:
  - "//streams/(main|dev|rel).*"
""")

        stream = p4.fetch_stream("-tmainline", "//streams/main")
        p4.save_stream(stream)

        stream = p4.fetch_stream("-tmainline", "//streams/dev1.0")
        p4.save_stream(stream)

        stream = p4.fetch_stream("-tmainline", "//streams/rel_1.0")
        p4.save_stream(stream)

        stream = p4.fetch_stream("-tmainline", "//streams/new")
        try:
            p4.save_stream(stream)
            self.assertTrue(False, "Expected exception not found")
        except P4.P4Exception as e:
            self.assertRegex(str(e), r"Stream name doesn't conform to valid regex format")


        # Test for invalid regex
        with open(self.config_path, "w") as f:
            f.write("""
msg_invalid_stream_name:
  - ""
  - "Stream name doesn't conform to valid regex format."
  - "See below for list of valid formats"

valid_stream_name_formats:
  - "*"
""")

        stream = p4.fetch_stream("-tmainline", "//streams/rel_2.0")
        try:
            p4.save_stream(stream)
            self.assertTrue(False, "Expected exception not found")
        except P4.P4Exception as e:
            self.assertRegex(str(e), r"Invalid config file for trigger")

if __name__ == '__main__':
    unittest.main()
