#!/bin/bash
set -u

declare ThisScript="${0##*/}"
declare Version=1.3.1
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare EdgeHost=syd-helix-04
declare -i LocalLogCount=0
declare -a LocalLogList
declare -i EdgeLogCount=0
declare -a EdgeLogList

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript v$Version]: Do parallel checkpoint on edge."

# Setup: Create site config dir if needed
mkdir -p /p4/common/site/config
ssh "$EdgeHost" mkdir -p /p4/common/site/config

msg "Configure serial checkpoint on commit and edge."

{
   echo export DO_PARALLEL_CHECKPOINTS=4
   echo KEEPCKPS=2
   echo KEEPJNLS=2
   echo KEEPCKPS=2
} > /p4/common/site/config/p4_1.vars.local

rsync /p4/common/site/config/p4_1.vars.local "$EdgeHost":/p4/common/site/config/p4_1.vars.local ||\
   errmsg "Failed to copy /p4/common/site/config/p4_1.vars.local."

msg "Request checkpoint on edge."
ssh "$EdgeHost" /p4/common/bin/request_replica_checkpoint.sh 1 ||\
   errmsg "Remote call to request_replica_checkpoint.sh indicated an error."

EdgeLogList[EdgeLogCount]="/p4/1/logs/request_checkpoint.log"
EdgeLogCount+=1

msg "Rotate journal on commit."
rotate_journal.sh 1 || errmsg "Error doing journal rotation."

LocalLogList[LocalLogCount]="/p4/1/logs/recreate_offline_db.log"
LocalLogCount+=1

msg "Recover checkpoint on edge."
ssh "$EdgeHost" /p4/common/bin/recreate_offline_db.sh 1 ||\
   errmsg "Remote call to recreate_offline_db.sh indicated an error."

EdgeLogList[EdgeLogCount]="/p4/1/logs/recreate_offline_db.log"
EdgeLogCount+=1

msg "Checking for syntax and unbound errors in scripts used in this test."
for log in "${LocalLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $log"
   grep ':unbound variable' "$log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $log"
   grep ':syntax error' "$log" && \
      errmsg "Found 'syntax error' in $log"
done

msg "Checking for syntax and unbound errors in scripts used in this test on edge host."
for log in "${EdgeLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
done

exit "$ErrorCount"
