package com.perforce.hws.server;

import java.util.regex.Pattern;

import org.apache.commons.lang3.ArrayUtils;

/**
 * Enum to help with the definition and building of REST paths.
 * 
 * All of our REST paths begin with a /api prefix followed by a
 * context which defines the responsibility for handling the
 * request and a version specifier for the api within this
 * context; currently there are HelixWebServices and Perforce
 * contexts.
 * 
 * Following the versioned context there are various fixed and 
 * dynamic components to a path which depend upon the operation 
 * being carried out.
 * 
 * Examples
 * --------
 * 
 * /api/hws/version       - get the current version of the hws api
 * /api/hws/v16.1/login   - login to hws using v16.1 of the api
 *  api/p4d/v16.1/localhost/branches - p4d branches in localhost
 */
public enum ApiPath {
	/** The generic '/api' path prefix. */
	Api("api"),
	/** A Perforce branch command processor. */
	Branch(":branch"),
	/** Route the request to a Perforce Branch command processor. */
	Branches("branches"),
	/** A Perforce change id, available in request params. */
	Change(":change"),
	/** Route the request to a Perforce Change command processor. */
	Changes("changes"),
	/** The Client id, available in request params. */
	Client(":client"),
	/** Route the request to a Perforce Workspace Client command processor. */
	Clients("clients"),
	/** The Command name, available in request params. */
	Command(":command"),
	/** Route the request to a Perforce Command command processor. */
	Commands("commands"),
	/** The hws configuration suite. */
	Config("config"),
	/** The Counter id, available in request params. */
	Counter(":counter"),
	/** Route the request to a Perforce Counter command processor. */
	Counters("counters"),
	/** The Depot id, available in request params. */
	Depot(":depot"),
	/** Route the request to a Perforce Depot command processor. */
	Depots("depots"),
	/** The internal documentation. */
	Doc("doc/index.html"),
	/** The internal documentation directory. */
	DocDir("doc"),
	/** Route the request to a Perforce Fixe command processor. */
	Fixes("fixes"),
	/** Route the request to a Perforce Git fusion repos. */
	GitFusionRepos("git-fusion-repos"),
	/** The Group id, available in request params. */
	Group(":group"),
	/** Route the request to a Perforce Group command processor. */
	Groups("groups"),
	/** The Hws context. */
	Hws("hws"),
	/** The Increment action for a counter. */
	Increment("increment"),
	/** The Job id, available in request params. */
	Job(":job"),
	/** Route the request to a Perforce Job command processor. */
	Jobs("jobs"),
	/** The Label id, available in request params. */
	Label(":label"),
	/** Route the request to a Perforce Label command processor. */
	Labels("labels"),
	/** The Login path, should be preceded by a context. */
	Login("login"),
	/** The P4d context. */
	P4d("p4d"),
	/** Route the request to list P4ds from the hws configuration. */
	P4ds("p4ds"),
	/** Route the request to a Perforce Path command processor. */
	Paths("paths"),
	/** The Platform version. */
	PlatformVersion(":platformVersion"),
	/** Route the request to a Perforce Protection command processor. */
	Protections("protections"),
	/** The git-fusion repo identfier, available in request params. */
	Repo(":repo"),
	/** The base path of the server. */
	Root(""),
	/** The Server id, available in request params. */
	Server(":server"),
	/** The Server id value, available in request params. */
	ServerId(":serverId"),
	/** Route the request to a Perforce Server command processor. */
	Servers("servers"),
	/** The Status of this HelixWebService. */
	Status("status"),
	/** The Stream id, available in request params. */
	Stream(":stream"),
	/** Route the request to a Perforce Stream command processor. */
	Streams("streams"),
	/** Route the request to a Perforce Trigger command processor. */
	Triggers("triggers"),
	/** The User id, available in request params. */
	User(":user"),
	/** Route the request to a Perforce User command processor. */
	Users("users"),
	/** The Version of a particular api. */
	Version("version");
	
	/** Pattern matching on version. */
	private static final String VERSION_PATTERN_STRING = "/v[0-9.]+/";
	
	/** The is login path for HWS. */
	public static final Pattern LOGIN_PATTERN = Pattern.compile(
    		"^/" + Api.path + "/" + Hws.path
    		+ VERSION_PATTERN_STRING + Login.path + ".*");
	
	/** The is login path for P4D. */
	public static final Pattern P4D_LOGIN_PATTERN = Pattern.compile(
    		"^/" + Api.path + "/" + P4d.path
    		+ VERSION_PATTERN_STRING + "[a-z0-9]+/" + Login.path + ".*");

    /** The status path pattern. */
    public static final Pattern STATUS_PATTERN = Pattern.compile(
    		"^/" + Api.path + "/" + Hws.path
    		+ VERSION_PATTERN_STRING + Status.path + "$");

	/** The path. */
	private String path;
	
	/**
	 * Instantiates a new api path.
	 *
	 * @param thePath the the path
	 */
	ApiPath(final String thePath) {
		this.path = thePath;
	}

	public String getPath() {
		return path;
	}
}
