package com.perforce.hws.server;

import org.apache.commons.lang3.ArrayUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.perforce.hws.server.ApiPath.PlatformVersion;
import static com.perforce.hws.server.ApiPath.Api;
import static com.perforce.hws.server.ApiPath.Hws;
import static com.perforce.hws.server.ApiPath.Server;
import static com.perforce.hws.server.ApiPath.P4d;

/**
 * Utility to build Api paths.
 */
public interface ApiPathUtils {

    /**
     * Gets the settings.
     *
     * @return the settings
     */
    HWSSettings getSettings();

    /**
     * Builds a path with the optional path prefix if set.
     *
     * @param path the path
     * @return the string
     */
    default String pathTo(String... path) {
        List<String> parts = new ArrayList<>();
        if (getSettings().getPrefix() != null) {
            parts.add(getSettings().getPrefix());
        }
        Collections.addAll(parts, path);
        String pathStr = parts.stream().collect(Collectors.joining("/"));
        if (!pathStr.startsWith("/")) {
            pathStr = "/" + pathStr;
        }
        return pathStr;
    }

    /**
     * Builds a string path.
     *
     * @param path the path
     * @return the string
     */
    default String pathTo(ApiPath... path) {
        String[] strPath =
                Arrays.stream(path).map(ApiPath::getPath)
                        .collect(Collectors.toList())
                        .toArray(new String[0]);

        return pathTo(strPath);
    }

    /**
     * Builds a path to api with the optional path prefix if set.
     *
     * @param path the path
     * @return the string
     */
    default String apiPathTo(final ApiPath ... path) {
        return pathTo(ArrayUtils.addAll(new ApiPath[] {Api}, path));
    }

    /**
     * Builds a path to Hws the optional path prefix if set.
     *
     * @param path the path
     * @return the string
     */
    default String hwsPathTo(final ApiPath ... path) {
        ApiPath[] prefix = new ApiPath[] {Hws, PlatformVersion};
        return apiPathTo(ArrayUtils.addAll(prefix, path));
    }

    /**
     * Builds a path to P4D the optional path prefix if set.
     *
     * @param path the path
     * @return the string
     */
    default String serverPathTo(final ApiPath ... path) {
        ApiPath[] prefix = new ApiPath[] {P4d, PlatformVersion, Server};
        return apiPathTo(ArrayUtils.addAll(prefix, path));
    }


    /**
     * Namespace version pattern.
     *
     * @return the pattern
     */
    default Pattern namespaceVersionPattern() {
        return Pattern.compile("^" + pathTo(Api) + "/[a-z0-9]+/version");
    }

    /**
     * Valid path pattern.
     *
     * @return the pattern
     */
    default Pattern validPathPattern() {
        return Pattern.compile("^" + pathTo(Api)
                + "/[a-z0-9]+/(?<version>v[.0-9]+)/?.*");
    }
}
