package com.perforce.hws.server;

import com.esotericsoftware.yamlbeans.YamlReader;
import com.esotericsoftware.yamlbeans.YamlReader.YamlReaderException;
import com.perforce.hwsclient.models.P4dConfigId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import spark.Request;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Application settings for Helix Web Services, initialized from a system
 * configuration file or request headers.
 * <p>
 * When adding settings, be sure to do a couple of things:
 * <p>
 * <ul>
 * <li>Document any configuration file key or per-request override settings on
 * the getter</li>
 * <li>Update the corresponding overrideFromSystemConfig()
 * and override() methods.</li>
 * </ul>
 */
public class HWSSettings implements Cloneable {
    /**
     * The maximum connections.
     */
    private static final int MAX_CONNECTIONS = 50;
    /**
     * The default hws port.
     */
    private static final int DEFAULT_PORT = 9000;
    /**
     * The Constant logger.
     */
    private static final Logger LOGGER =
            LoggerFactory.getLogger(HWSSettings.class);

    /**
     * The hws port.
     */
    private int hwsPort = DEFAULT_PORT;

    /**
     * The access control allow headers.
     */
    private String accessControlAllowHeaders = "*";

    /**
     * The access control allow origin.
     */
    private String accessControlAllowOrigin = "*";

    /**
     * The access control request method.
     */
    private String accessControlRequestMethod = "*";

    /**
     * The auto trust.
     */
    private boolean autoTrust = false;

    /**
     * The command whitelist.
     */
    private List<WhitelistedCommand> commandWhitelist;

    /**
     * The enable git fusion.
     */
    private boolean enableGitFusion = false;

    /**
     * The enable man in middle attacks.
     */
    private boolean enableManInMiddleAttacks = false;

    /**
     * The enable https.
     */
    private boolean enableHttps = false;

    /**
     * The git fusion depot.
     */
    private String gitFusionDepot = ".git-fusion";

    /**
     * The git fusion create config description format.
     */
    private String gitFusionCreateConfigDescriptionFormat
            = "Creating git fusion repo %s";

    /**
     * The git fusion delete all keys description format.
     */
    private String gitFusionDeleteAllKeysDescriptionFormat
            = "Deleting all SSH keys for user %s";

    /**
     * The git fusion delete key description format.
     */
    private String gitFusionDeleteKeyDescriptionFormat
            = "Deleting SSH key %s for user %s";

    /**
     * The git fusion replace key description format.
     */
    private String gitFusionReplaceKeyDescriptionFormat
            = "Replacing SSH key %s for user %s";

    /**
     * The jwt signing key.
     */
    private String jwtSigningKey;

    /**
     * The jwt timeout in seconds.
     */
    private int jwtTimeoutInSeconds = (int) TimeUnit.DAYS.toSeconds(2);

    /**
     * The keystore file.
     */
    private String keystoreFile;

    /**
     * The keystore password.
     */
    private String keystorePassword;

    /**
     * The p4d config dir.
     */
    private String p4dConfigDir = null;

    /**
     * The server "prefix" path.
     *
     * In case it's been mounted as a servlet in another Server container. You really
     * won't need this in the default deployment.
     */
    private String prefix = null;

    /**
     * The request filter path.
     */
    private String requestFilterPath = null;

    /**
     * The setumask.
     */
    private int setumask = 0;

    /**
     * The setuid.
     */
    private int setuid = 0;

    /**
     * The setgid.
     */
    private int setgid = 0;

    /**
     * The system config path.
     */
    // TODO need Unix vs Windows defaults
    private String systemConfigPath = "/etc/perforce/helix_web_services.conf";

    /**
     * The trust fingerprints.
     */
    private String trustFingerprints = null;

    /**
     * The truststore file.
     */
    private String truststoreFile = null;

    /**
     * The truststore password.
     */
    private String truststorePassword = null;

    /**
     * The web hooks.
     */
    private List<WebHook> webHooks = null;

    /**
     * The p4d config map.
     */
    private Map<String, P4dConfig> p4dConfigMap = null;

    /**
     * The default api level.
     */
    private int defaultApiLevel;

    /**
     * The workspace dir.
     */
    // TODO need Unix vs Windows defaults
    private String workspaceDir = "/var/lib/perforce/helix_web_services/workspaces";

    // The total number of Server connections to allow by this machine per
    /**
     * The max server connections.
     */
    // P4PORT
    private int maxServerConnections = MAX_CONNECTIONS;

    /**
     * The Constant RE_P4_API_LEVEL.
     */
    // All P4 service commands should specify API level via the path.
    private static final Pattern RE_P4_API_LEVEL =
            Pattern.compile("^/p4/v(?<api>\\d+)/.*$");

    /**
     * The auth p4d.  The auth P4d is the
     * p4d configuration to be used for the login
     * api when a server is not specified.
     */
    private String authP4d;

    /**
     * Instantiates a new HWS settings.
     */
    public HWSSettings() {
        commandWhitelist = new ArrayList<>();
        commandWhitelist.add(new WhitelistedCommand("info"));
        commandWhitelist.add(new WhitelistedCommand("files", "-m"));

        webHooks = new ArrayList<>();
    }

    /**
     * Override from environment.
     */
    public void overrideFromEnvironment() {
        checkEnvAndOverride(ConfigurationKey.AUTO_TRUST,
                (s) -> setAutoTrust(Boolean.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.ENABLE_GIT_FUSION,
                (s) -> setEnableGitFusion(Boolean.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.ENABLE_HTTPS,
                (s) -> setEnableHttps(Boolean.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.ENABLE_MAN_IN_MIDDLE_ATTACKS,
                (s) -> setEnableManInMiddleAttacks(Boolean.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.GITFUSIONDEPOT,
                this::setGitFusionDepot);
        checkEnvAndOverride(ConfigurationKey.HWS_PORT,
                (s) -> setHwsPort(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.JWT_SIGNING_KEY,
                this::setJwtSigningKey);
        checkEnvAndOverride(ConfigurationKey.JWT_TIMEOUT_IN_SECONDS,
                (s) -> setJwtTimeoutInSeconds(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.KEYSTORE_FILE, this::setKeystoreFile);
        checkEnvAndOverride(ConfigurationKey.KEYSTORE_PASSWORD,
                this::setKeystorePassword);
        checkEnvAndOverride(ConfigurationKey.MAX_SERVER_CONNECTIONS,
                (s) -> setMaxServerConnections(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.P4DCONFIGDIR, this::setP4dConfigDir);
        checkEnvAndOverride(ConfigurationKey.PREFIX, this::setPrefix);
        checkEnvAndOverride(ConfigurationKey.REQUEST_FILTER_PATH,
                this::setRequestFilterPath);
        checkEnvAndOverride(ConfigurationKey.SETUMASK,
                (s) -> setSetumask(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.SETGID,
                (s) -> setSetgid(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.SETUID,
                (s) -> setSetuid(Integer.valueOf(s)));
        checkEnvAndOverride(ConfigurationKey.SYSTEM_CONFIG_PATH,
                this::setSystemConfigPath);
        checkEnvAndOverride(ConfigurationKey.TRUSTSTORE_FILE,
                this::setTruststoreFile);
        checkEnvAndOverride(ConfigurationKey.TRUSTSTORE_PASSWORD,
                this::setTruststorePassword);
        checkEnvAndOverride(ConfigurationKey.TRUST_FINGERPRINTS,
                this::setTrustFingerprints);
        checkEnvAndOverride(ConfigurationKey.WORKSPACE_DIR, this::setWorkspaceDir);
        checkEnvAndOverride(ConfigurationKey.HWS_AUTH_P4D, this::setAuthP4d);
    }

    /**
     * Uses the systemConfigPath property to read a file (if it exists) and
     * will override properties defined there.
     *
     * <p>This uses <em>documented</em> property names from the config file, mostly
     * to mirror usage from the original implementation. There is no "automagic"
     * bean mapping.
     *
     * @throws IOException Signals that an I/O exception has occurred.
     */
    public void overrideFromSystemConfig() throws IOException {
        if (!new File(systemConfigPath).exists()) {
            return;
        }

        try (FileReader fileReader = new FileReader(systemConfigPath)) {
            YamlReader yamlReader = new YamlReader(fileReader);
            @SuppressWarnings("unchecked")
            Map<String, Object> config = (Map<String, Object>) yamlReader.read();
            checkAndOverride(config, ConfigurationKey.ACCESS_CONTROL_ALLOW_ORIGIN,
                    this::setAccessControlAllowOrigin);
            checkAndOverride(config, ConfigurationKey.ACCESS_CONTROL_ALLOW_HEADERS,
                    this::setAccessControlAllowHeaders);
            checkAndOverride(config, ConfigurationKey.ACCESS_CONTROL_REQUEST_METHOD,
                    this::setAccessControlRequestMethod);
            checkAndOverride(config, ConfigurationKey.AUTO_TRUST,
                    (s) -> setAutoTrust(Boolean.valueOf(s)));
            checkForWhitelist(config);
            checkAndOverride(config, ConfigurationKey.DEFAULT_API_LEVEL,
                    (s) -> setDefaultApiLevel(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.ENABLE_GIT_FUSION,
                    (s) -> setEnableGitFusion(Boolean.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.ENABLE_HTTPS,
                    (s) -> setEnableHttps(Boolean.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.ENABLE_MAN_IN_MIDDLE_ATTACKS,
                    (s) -> setEnableManInMiddleAttacks(Boolean.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.GITFUSIONDEPOT,
                    this::setGitFusionDepot);
            checkAndOverride(config, ConfigurationKey.HWS_PORT,
                    (s) -> setHwsPort(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.JWT_SIGNING_KEY,
                    this::setJwtSigningKey);
            checkAndOverride(config, ConfigurationKey.JWT_TIMEOUT_IN_SECONDS,
                    (s) -> setJwtTimeoutInSeconds(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.KEYSTORE_FILE,
                    this::setKeystoreFile);
            checkAndOverride(config, ConfigurationKey.KEYSTORE_PASSWORD,
                    this::setKeystorePassword);
            checkAndOverride(config, ConfigurationKey.MAX_SERVER_CONNECTIONS,
                    (s) -> setMaxServerConnections(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.P4DCONFIGDIR,
                    this::setP4dConfigDir);
            checkAndOverride(config, ConfigurationKey.PREFIX,
                    this::setPrefix);
            checkAndOverride(config, ConfigurationKey.REQUEST_FILTER_PATH,
                    this::setRequestFilterPath);
            checkAndOverride(config, ConfigurationKey.SETUMASK,
                    (s) -> setSetumask(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.SETGID,
                    (s) -> setSetgid(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.SETUID,
                    (s) -> setSetuid(Integer.valueOf(s)));
            checkAndOverride(config, ConfigurationKey.TRUSTSTORE_FILE,
                    this::setTruststoreFile);
            checkAndOverride(config, ConfigurationKey.TRUSTSTORE_PASSWORD,
                    this::setTruststorePassword);
            checkAndOverride(config, ConfigurationKey.TRUST_FINGERPRINTS,
                    this::setTrustFingerprints);
            checkForWebHooks(config);
            checkAndOverride(config, ConfigurationKey.WORKSPACE_DIR,
                    this::setWorkspaceDir);
            checkAndOverride(config, ConfigurationKey.HWS_AUTH_P4D,
                    this::setAuthP4d);
        }
    }

    /**
     * Override all values on a per-request basis.
     * <p>
     * What happens is that we read in a single instance of HWSSettings, and
     * set various options from a config file. Then, on each request, we clone
     * the settings object and look for header overrides.
     * <p>
     * Not all values are overridable.
     * <p>
     * Also, keys are slightly different than might show up elsewhere. Web
     * servers and other application frameworks like to tweak case and block
     * things like underscores. So don't use underscores and put everything
     * in uppercase as a convention.
     *
     * @param req the req
     */
    public void override(final Request req) {
        checkAndOverride(req, ConfigurationKey.GITFUSIONDEPOT,
                this::setGitFusionDepot);
    }

    /**
     * If the provided key exists in the map, call the consumer.
     * <p>
     * The map here is probably read in from the Yaml file.
     *
     * @param config   the config
     * @param ck       the key
     * @param consumer the consumer
     */
    private void checkAndOverride(final Map<String, Object> config,
                                  final ConfigurationKey ck,
                                  final Consumer<String> consumer) {
        String key = ck.toString();
        if (config.containsKey(key)) {
            consumer.accept((String) config.get(key));
        }
    }

    /**
     * If the provided key exists in the map, call the consumer.
     * <p>
     * The map here is probably read in from the Yaml file.
     *
     * @param ck       the configuration key
     * @param consumer the consumer
     */
    private void checkEnvAndOverride(final ConfigurationKey ck,
                                     final Consumer<String> consumer) {
        String key = ck.toString();
        if (System.getenv(key) != null) {
            consumer.accept(System.getenv(key));
        }
        if (System.getProperty(key) != null) {
            consumer.accept(System.getProperty(key));
        }
    }

    /**
     * If the keySuffix, appended to our standard header prefix, exists on the
     * request, call the consumer.
     * <p>
     * Unlike our "map" method, the consumer here must consume strings.
     *
     * @param req      the req
     * @param ck       the key suffix
     * @param consumer the consumer
     */
    private void checkAndOverride(final Request req,
                                  final ConfigurationKey ck,
                                  final Consumer<String> consumer) {
        String header = String.format(
                "X-Perforce-Helix-Web-Services-%s", ck.toString());
        if (req.headers(header) != null) {
            consumer.accept(req.headers(header));
        }
    }

    /**
     * Check for whitelist.
     *
     * @param config the config
     */
    private void checkForWhitelist(final Map<String, Object> config) {
        String key = ConfigurationKey.COMMAND_WHITELIST.toString();
        if (config.containsKey(key)) {
            List<WhitelistedCommand> newList = new ArrayList<>();
            @SuppressWarnings("unchecked")
            List<Object> whitelist = (List<Object>) config.get(key);
            whitelist.forEach((item) -> {
                if (item instanceof List) {
                    @SuppressWarnings("unchecked")
                    List<String> options = (List<String>) item;
                    String command = options.get(0);
                    List<String> args = options.subList(1, options.size());
                    newList.add(new WhitelistedCommand(command,
                            args.toArray(new String[options.size()])));
                } else {
                    String command = (String) item;
                    newList.add(new WhitelistedCommand(command));
                }
            });
            commandWhitelist = newList;
        }
    }

    /**
     * Check for web hooks.
     *
     * @param config the config
     */
    @SuppressWarnings("unchecked")
    private void checkForWebHooks(final Map<String, Object> config) {
        String key = ConfigurationKey.WEBHOOKS.toString();
        if (config.containsKey(key)) {
            @SuppressWarnings("rawtypes")
            List hookDefs = (List) config.get(key);
            hookDefs.forEach(hookDef -> {
                Map<String, Object> map = (Map<String, Object>) hookDef;
                String id = (String) map.get("id");
                String url = (String) map.get("url");
                Boolean disableSslValidation = false;
                if (map.containsKey("disableSslValidation")) {
                    disableSslValidation =
                            Boolean.valueOf(
                                    (String) map.get("disableSslValidation"));
                }
                webHooks.add(new WebHook(id, url, disableSslValidation));
            });
        }
    }


    //-----------------------------------------------------------------------
    // Getters and setters
    //-----------------------------------------------------------------------
    // Please clearly indicate if this method can be overridden in a request,
    // and what the value we read from our system config is.

    /**
     * The port our HTTP server responds to.
     * <p>
     * Defaults to 9000.
     * <p>
     * Config file property: <code>HWS_PORT</code>
     *
     * @return the hws port
     */
    public int getHwsPort() {
        return hwsPort;
    }

    /**
     * Sets the hws port.
     *
     * @param hwsPort the new hws port
     */
    public void setHwsPort(final int hwsPort) {
        this.hwsPort = hwsPort;
    }

    /**
     * Sets the CORS Response header Access-Control-Allow-Headers.
     *
     * @return the access control allow headers
     */
    public String getAccessControlAllowHeaders() {
        return accessControlAllowHeaders;
    }

    /**
     * Sets the access control allow headers.
     *
     * @param accessControlAllowHeaders the new access control allow headers
     */
    public void setAccessControlAllowHeaders(
            final String accessControlAllowHeaders) {
        this.accessControlAllowHeaders = accessControlAllowHeaders;
    }

    /**
     * Sets the CORS Response header Access-Control-Allow-Origin.
     *
     * @return the access control allow origin
     */
    public String getAccessControlAllowOrigin() {
        return accessControlAllowOrigin;
    }

    /**
     * Sets the access control allow origin.
     *
     * @param accessControlAllowOrigin the new access control allow origin
     */
    public void setAccessControlAllowOrigin(final String accessControlAllowOrigin) {
        this.accessControlAllowOrigin = accessControlAllowOrigin;
    }

    /**
     * Sets the CORS Response header Access-Control-Request-Method.
     *
     * @return the access control request method
     */
    public String getAccessControlRequestMethod() {
        return accessControlRequestMethod;
    }

    /**
     * Sets the access control request method.
     *
     * @param accessControlRequestMethod the new access control request method
     */
    public void setAccessControlRequestMethod(
            final String accessControlRequestMethod) {
        this.accessControlRequestMethod = accessControlRequestMethod;
    }

    /**
     * Use the "p4 trust -y" flag to trust new P4 Servers.<p/>
     * <p>
     * Configuration file property: <code>AUTO_TRUST</code>
     *
     * @return true, if is auto trust
     */
    public boolean isAutoTrust() {
        return autoTrust;
    }

    /**
     * Sets the auto trust.
     *
     * @param autoTrust the new auto trust
     */
    public void setAutoTrust(final boolean autoTrust) {
        this.autoTrust = autoTrust;
    }

    /**
     * Allowed p4 commands for the <code>/p4/v[api]/commands</code> methods.
     * <p>
     * Configuration file property: <code>COMMAND_WHITELIST</code>.
     *
     * @return the command whitelist
     */
    public List<WhitelistedCommand> getCommandWhitelist() {
        return commandWhitelist;
    }

    /**
     * Sets the command whitelist.
     *
     * @param commandWhitelist the new command whitelist
     */
    public void setCommandWhitelist(
            final List<WhitelistedCommand> commandWhitelist) {
        this.commandWhitelist = commandWhitelist;
    }

    /**
     * If the application should respond to git fusion requests.<p/>
     * <p>
     * If your Perforce servers do not run git fusion, you probably want to
     * return 404s, to any application that tries to interact with the server.
     * <p>
     * Configuration file property: <code>ENABLE_GIT_FUSION</code>
     *
     * @return true, if is enable git fusion
     */
    public boolean isEnableGitFusion() {
        return enableGitFusion;
    }

    /**
     * Sets the enable git fusion.
     *
     * @param enableGitFusion the new enable git fusion
     */
    public void setEnableGitFusion(final boolean enableGitFusion) {
        this.enableGitFusion = enableGitFusion;
    }

    /**
     * Gets the default api level.
     *
     * @return the default api level
     */
    public int getDefaultApiLevel() {
        return defaultApiLevel;
    }

    /**
     * Sets the default api level.
     *
     * @param defaultApiLevel the new default api level
     */
    public void setDefaultApiLevel(final int defaultApiLevel) {
        this.defaultApiLevel = defaultApiLevel;
    }

    /**
     * Tell the server to always trust all TLS connections to p4d.<p/>
     * <p>
     * Note: this property should not be documented in user guides. This is to
     * handle testing scenarios, where we generate p4ds on the fly with SSL
     * enabled.<p/>
     * <p>
     * Configuration file property: <code>ENABLE_MAN_IN_MIDDLE_ATTACKS</code>
     *
     * @return true, if is enable man in middle attacks
     */
    public boolean isEnableManInMiddleAttacks() {
        return enableManInMiddleAttacks;
    }

    /**
     * Sets the enable man in middle attacks.
     *
     * @param enableManInMiddleAttacks the new enable man in middle attacks
     */
    public void setEnableManInMiddleAttacks(final boolean enableManInMiddleAttacks) {
        this.enableManInMiddleAttacks = enableManInMiddleAttacks;
    }

    /**
     * If true, we enable HTTPS for the underlying Jetty server.
     * <p>
     * Configuration file property: <code>ENABLE_HTTPS</code>
     *
     * @return true, if is enable https
     */
    public boolean isEnableHttps() {
        return enableHttps;
    }

    /**
     * Sets the enable https.
     *
     * @param enableHttps the new enable https
     */
    public void setEnableHttps(final boolean enableHttps) {
        this.enableHttps = enableHttps;
    }

    /**
     * The name of the git fusion depot on the Perforce server.
     * <p>
     * Configuration file property: <code>GIT_FUSION_DEPOT</code>
     * <p>
     * Request header override:
     * <code>X-Perforce-Helix-Web-Services-GITFUSIONDEPOT</code>
     *
     * @return the git fusion depot
     */
    public String getGitFusionDepot() {
        return gitFusionDepot;
    }

    /**
     * Sets the git fusion depot.
     *
     * @param gitFusionDepot the new git fusion depot
     */
    public void setGitFusionDepot(final String gitFusionDepot) {
        this.gitFusionDepot = gitFusionDepot;
    }

    /**
     * Gets the git fusion create config description format.
     *
     * @return the git fusion create config description format
     */
    public String getGitFusionCreateConfigDescriptionFormat() {
        return gitFusionCreateConfigDescriptionFormat;
    }

    /**
     * Sets the git fusion create config description format.
     *
     * @param gitFusionCreateConfigDescriptionFormat the new
     *        git fusion create config description format
     */
    public void setGitFusionCreateConfigDescriptionFormat(
            final String gitFusionCreateConfigDescriptionFormat) {
        this.gitFusionCreateConfigDescriptionFormat =
                gitFusionCreateConfigDescriptionFormat;
    }

    /**
     * Gets the git fusion delete all keys description format.
     *
     * @return the git fusion delete all keys description format
     */
    public String getGitFusionDeleteAllKeysDescriptionFormat() {
        return gitFusionDeleteAllKeysDescriptionFormat;
    }

    /**
     * Sets the git fusion delete all keys description format.
     *
     * @param gitFusionDeleteAllKeysDescriptionFormat the new
     *        git fusion delete all keys description format
     */
    public void setGitFusionDeleteAllKeysDescriptionFormat(
            final String gitFusionDeleteAllKeysDescriptionFormat) {
        this.gitFusionDeleteAllKeysDescriptionFormat =
                gitFusionDeleteAllKeysDescriptionFormat;
    }

    /**
     * Gets the git fusion delete key description format.
     *
     * @return the git fusion delete key description format
     */
    public String getGitFusionDeleteKeyDescriptionFormat() {
        return gitFusionDeleteKeyDescriptionFormat;
    }

    /**
     * Sets the git fusion delete key description format.
     *
     * @param gitFusionDeleteKeyDescriptionFormat the new git
     *        fusion delete key description format
     */
    public void setGitFusionDeleteKeyDescriptionFormat(
            final String gitFusionDeleteKeyDescriptionFormat) {
        this.gitFusionDeleteKeyDescriptionFormat =
                gitFusionDeleteKeyDescriptionFormat;
    }

    /**
     * Gets the git fusion replace key description format.
     *
     * @return the git fusion replace key description format
     */
    public String getGitFusionReplaceKeyDescriptionFormat() {
        return gitFusionReplaceKeyDescriptionFormat;
    }

    /**
     * Sets the git fusion replace key description format.
     *
     * @param gitFusionReplaceKeyDescriptionFormat the new
     *        git fusion replace key description format
     */
    public void setGitFusionReplaceKeyDescriptionFormat(
            final String gitFusionReplaceKeyDescriptionFormat) {
        this.gitFusionReplaceKeyDescriptionFormat =
                gitFusionReplaceKeyDescriptionFormat;
    }

    /**
     * A base64-encoded key used to sign tokens for authentication.
     * <p>
     * Configuration file property: <code>JWT_SIGNING_KEY</code>
     *
     * @return The signing key to use.
     */
    public String getJwtSigningKey() {
        return jwtSigningKey;
    }

    /**
     * Sets the jwt signing key.
     *
     * @param jwtSigningKey the new jwt signing key
     */
    public void setJwtSigningKey(final String jwtSigningKey) {
        this.jwtSigningKey = jwtSigningKey;
    }

    /**
     * If > 0, we'll generate an expiration for each JWT token.
     * <p>
     * Configuration file property: <code>JWT_TIMEOUT_IN_SECONDS</code>
     *
     * @return The current expiration to use.
     */
    public int getJwtTimeoutInSeconds() {
        return jwtTimeoutInSeconds;
    }

    /**
     * Sets the jwt timeout in seconds.
     *
     * @param jwtTimeoutInSeconds the new jwt timeout in seconds
     */
    public void setJwtTimeoutInSeconds(final int jwtTimeoutInSeconds) {
        this.jwtTimeoutInSeconds = jwtTimeoutInSeconds;
    }

    /**
     * For secure servers, the keystore file location.
     *
     * @return the keystore file
     */
    public String getKeystoreFile() {
        return keystoreFile;
    }

    /**
     * Sets the keystore file.
     *
     * @param keystoreFile the new keystore file
     */
    public void setKeystoreFile(final String keystoreFile) {
        this.keystoreFile = keystoreFile;
    }

    /**
     * For secure servers, the password to access the keystore file (nullable).
     *
     * @return the keystore password
     */
    public String getKeystorePassword() {
        return keystorePassword;
    }

    /**
     * Sets the keystore password.
     *
     * @param keystorePassword the new keystore password
     */
    public void setKeystorePassword(final String keystorePassword) {
        this.keystorePassword = keystorePassword;
    }

    /**
     * The local directory where "p4d config" files are located.
     * <p>
     * Each file be named after the id value.
     * <p>
     * Configuration file property: <code>P4DCONFIGDIR</code>
     * </p>
     *
     * @return the p4d config dir
     */
    public String getP4dConfigDir() {
        return p4dConfigDir;
    }

    /**
     * Sets the p4d config dir.
     *
     * @param p4dConfigDir the new p4d config dir
     */
    public void setP4dConfigDir(final String p4dConfigDir) {
        this.p4dConfigDir = p4dConfigDir;
    }

    /**
     * Gets the prefix.
     *
     * @return the prefix
     */
    public String getPrefix() {
        return prefix;
    }

    /**
     * Sets the prefix.
     *
     * @param prefix the new prefix
     */
    public void setPrefix(final String prefix) {
        this.prefix = prefix;
    }

    /**
     * Gets the re p4 api level.
     *
     * @return the re p4 api level
     */
    public static Pattern getReP4ApiLevel() {
        return RE_P4_API_LEVEL;
    }

    /**
     * The directory where we create temporary p4 client workspaces.
     * <p>
     * Configuration file property: <code>WORKSPACE_DIR</code>
     *
     * @return the workspace dir
     */
    public String getWorkspaceDir() {
        return workspaceDir;
    }

    /**
     * Sets the workspace dir.
     *
     * @param workspaceDir the new workspace dir
     */
    public void setWorkspaceDir(final String workspaceDir) {
        this.workspaceDir = workspaceDir;
    }

    /**
     * Where we load the helix_web_services.conf YAML file.
     *
     * @return the system config path
     */
    public String getSystemConfigPath() {
        return systemConfigPath;
    }

    /**
     * Sets the system config path.
     *
     * @param systemConfigPath the new system config path
     */
    public void setSystemConfigPath(final String systemConfigPath) {
        this.systemConfigPath = systemConfigPath;
    }

    /**
     * The maximum number of server connections we allow per P4PORT. Defaults to 50.
     * <p>
     * Any change in this value should require a server restart.
     * <p>
     * Configuration file property: <code>MAX_SERVER_CONNECTIONS</code>
     *
     * @return The number of connections per P4PORT. To disable, set to -1.
     */
    public int getMaxServerConnections() {
        return maxServerConnections;
    }

    /**
     * Sets the max server connections.
     *
     * @param maxServerConnections the new max server connections
     */
    public void setMaxServerConnections(final int maxServerConnections) {
        this.maxServerConnections = maxServerConnections;
    }

    /**
     * If set, returns a path to a javaScript file where you can inspect the
     * request and fail it if it doesn't have valid parameters.
     * <p>
     * This should be a script that defines a function called validateRequest
     * that accepts the spark Request object as a parameter.
     * <p>
     * Configuration file property: <code>REQUEST_FILTER_PATH</code>
     *
     * @return the request filter path
     */
    public String getRequestFilterPath() {
        return requestFilterPath;
    }

    /**
     * Sets the request filter path.
     *
     * @param requestFilterPath the new request filter path
     */
    public void setRequestFilterPath(final String requestFilterPath) {
        this.requestFilterPath = requestFilterPath;
    }

    /**
     * Gets the setumask.
     *
     * @return the setumask
     */
    public int getSetumask() {
        return setumask;
    }

    /**
     * Sets the setumask.
     *
     * @param setumask the new setumask
     */
    public void setSetumask(final int setumask) {
        this.setumask = setumask;
    }

    /**
     * Gets the setuid.
     *
     * @return the setuid
     */
    public int getSetuid() {
        return setuid;
    }

    /**
     * Sets the setuid.
     *
     * @param setuid the new setuid
     */
    public void setSetuid(final int setuid) {
        this.setuid = setuid;
    }

    /**
     * Gets the setgid.
     *
     * @return the setgid
     */
    public int getSetgid() {
        return setgid;
    }

    /**
     * Sets the setgid.
     *
     * @param setgid the new setgid
     */
    public void setSetgid(final int setgid) {
        this.setgid = setgid;
    }

    /**
     * If set, we will always trust the SSL fingerprints in this file.
     * <p>
     * This is likely going to only be used by installations like Helix Cloud.
     * <p>
     * Configuration file property: <code>TRUST_FINGERPRINTS</code>
     *
     * @return the trust fingerprints
     */
    public String getTrustFingerprints() {
        return trustFingerprints;
    }

    /**
     * Sets the trust fingerprints.
     *
     * @param trustFingerprints the new trust fingerprints
     */
    public void setTrustFingerprints(final String trustFingerprints) {
        this.trustFingerprints = trustFingerprints;
    }

    /**
     * For secure servers, the truststore file location.
     * <p>
     * If empty, we reuse the keystore file location.
     *
     * @return the truststore file
     */
    public String getTruststoreFile() {
        return truststoreFile;
    }

    /**
     * Sets the truststore file.
     *
     * @param truststoreFile the new truststore file
     */
    public void setTruststoreFile(final String truststoreFile) {
        this.truststoreFile = truststoreFile;
    }

    /**
     * Password to access the truststore file.
     *
     * @return the truststore password
     */
    public String getTruststorePassword() {
        return truststorePassword;
    }

    /**
     * Sets the truststore password.
     *
     * @param truststorePassword the new truststore password
     */
    public void setTruststorePassword(final String truststorePassword) {
        this.truststorePassword = truststorePassword;
    }

    /**
     * Gets the web hooks.
     *
     * @return the web hooks
     */
    public List<WebHook> getWebHooks() {
        return webHooks;
    }

    /**
     * Sets the web hooks.
     *
     * @param webHooks the new web hooks
     */
    public void setWebHooks(final List<WebHook> webHooks) {
        this.webHooks = webHooks;
    }

    /**
     * Checks for web hook.
     *
     * @param id the id
     * @return true, if successful
     */
    public boolean hasWebHook(final String id) {
        return webHooks.stream().anyMatch(h -> h.getId().equals(id));
    }

    /**
     * Gets the web hook.
     *
     * @param id the id
     * @return the web hook
     */
    public WebHook getWebHook(final String id) {
        return webHooks.stream().filter(h -> h.getId().equals(id)).findFirst().get();
    }

    /**
     * Gets the p4d config map.
     *
     * @return the p4d config map
     */
    public Map<String, P4dConfig> getP4dConfigMap() {
        if (p4dConfigMap == null) {
            resetP4dConfigMap();
        }
        return p4dConfigMap;
    }

    /**
     * Reset p4d config map.
     */
    public void resetP4dConfigMap() {
        List<P4dConfig> p4dConfigs = listP4dConfig(p4dConfigDir);
        p4dConfigMap = p4dConfigs.stream().collect(
                Collectors.toMap(P4dConfigId::getId, x -> x));
    }

    /**
     * List p4d config.
     *
     * @param configDir the config dir
     * @return the list
     */
    private List<P4dConfig> listP4dConfig(final String configDir) {
        if (configDir == null) {
            throw new IllegalStateException(
                    "Configuration for "
                            + ConfigurationKey.P4DCONFIGDIR
                            + " has not been set");
        }
        LOGGER.info("Reading P4D configurations from " + configDir);
        try {
            Path configPath = Paths.get(configDir);
            if (!Files.exists(configPath)) {
                return Collections.emptyList();
            }
            return Files.list(Paths.get(configDir))
                    .map(p -> loadP4dConfig(p))
                    .filter(p -> p != null)
                    .collect(Collectors.toList());
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Load p4d config for the path. If the file is empty or
     * is invalid we simply log and continue.
     *
     * @param path the path
     * @return the p4d config
     */
    private P4dConfig loadP4dConfig(final Path path) {
        // Just log any issues, we do not want to break
        // server startup because of a bad file
        P4dConfig config = null;
        try (FileReader fileReader = new FileReader(path.toFile())) {
            YamlReader yamlReader = new YamlReader(fileReader);
            config = yamlReader.read(P4dConfig.class);
            if (config == null) {
                LOGGER.error("No configuration found in file " + path.toString());
            } else {
            	LOGGER.info("Config for server id " + config.getId()
            				+ " loaded successfully from "
            				+ path.toString());
            }
        } catch (YamlReaderException e) {
            LOGGER.error("Invalid configuration found in file "
                    + path.toString(), e);
        } catch (IOException e) {
            LOGGER.error("Configuration file error " + path.toString(), e);
        }
        return config;
    }

    /**
     * Gets the p4d config ids.
     *
     * @return the p4d config ids
     */
    public List<P4dConfigId> getP4dConfigIds() {
        return getP4dConfigMap().entrySet().stream()
                .map(e -> e.getValue())
                .map(c -> {
                    P4dConfigId id = new P4dConfigId();
                    id.setDescription(c.getDescription());
                    id.setId(c.getId());
                    id.setName(c.getName());
                    return id;
                }).collect(Collectors.toList());
    }

    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        HWSSettings c = (HWSSettings) super.clone();

        // Watch out, while Strings are immutable, mutable properties should
        // be duplicated.
        c.commandWhitelist = new ArrayList<>(c.commandWhitelist);

        return c;
    }

    /**
     * Gets the auth p4d. The auth P4d is the
     * p4d configuration to be used for the login
     * api when a server is not specified.
     *
     * @return the auth p4d
     */
    public String getAuthP4d() {
        return authP4d;
    }

    /**
     * Sets the auth p4d. The auth P4d is the
     * p4d configuration to be used for the login
     * api when a server is not specified.
     *
     * @param authP4d the new auth p4d
     */
    public void setAuthP4d(final String authP4d) {
        this.authP4d = authP4d;
    }

    /**
     * Fun little container structure for indicating commands we allow users
     * to run via our "run a generic command API".
     */
    public static class WhitelistedCommand {

        /**
         * The command.
         */
        private String command;

        /**
         * The required args.
         */
        private List<String> requiredArgs = null;

        /**
         * Instantiates a new whitelisted command.
         *
         * @param command      the command
         * @param requiredArgs the required args
         */
        public WhitelistedCommand(final String command,
                                  final String... requiredArgs) {
            this.command = command;
            if (requiredArgs != null && requiredArgs.length > 0) {
                this.requiredArgs = Arrays.asList(requiredArgs);
            }
        }

        /**
         * The name of the command. <p/>
         * <p>
         * If you want aliases to be matched, you must list each alias explicitly.
         *
         * @return the command
         */
        public String getCommand() {
            return command;
        }

        /**
         * Sets the command.
         *
         * @param command the new command
         */
        public void setCommand(final String command) {
            this.command = command;
        }

        /**
         * Optional list of args we must use whenever this command is called.<p/>
         *
         * @return If not set, is null.
         */
        public List<String> getRequiredArgs() {
            return requiredArgs;
        }

        /**
         * Sets the required args.
         *
         * @param requiredArgs the new required args
         */
        public void setRequiredArgs(final List<String> requiredArgs) {
            this.requiredArgs = requiredArgs;
        }

        /* (non-Javadoc)
         * @see java.lang.Object#equals(java.lang.Object)
         */
        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            WhitelistedCommand that = (WhitelistedCommand) o;

            if (!command.equals(that.command)) {
                return false;
            }
            return !(requiredArgs != null
                    ? !requiredArgs.equals(
                    that.requiredArgs) : that.requiredArgs != null);

        }

        /* (non-Javadoc)
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            int result = command.hashCode();
            final int prime = 31;
            result = prime * result + (requiredArgs != null
                    ? requiredArgs.hashCode() : 0);
            return result;
        }

        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "WhitelistedCommand{"
                    + "command='" + command + '\''
                    + ", requiredArgs=" + requiredArgs
                    + '}';
        }
    }

    /**
     * The Class WebHook.
     */
    public static class WebHook {

        /**
         * The id.
         */
        private String id;

        /**
         * The url.
         */
        private String url;

        /**
         * The disable ssl validation.
         */
        private boolean disableSslValidation;

        /**
         * Instantiates a new web hook.
         *
         * @param id                   the id
         * @param url                  the url
         * @param disableSslValidation the disable ssl validation
         */
        public WebHook(final String id, final String url,
                       final boolean disableSslValidation) {
            this.id = id;
            this.url = url;
            this.disableSslValidation = disableSslValidation;
        }

        /**
         * Instantiates a new web hook.
         */
        public WebHook() {
        }

        /**
         * Gets the id.
         *
         * @return the id
         */
        public String getId() {
            return id;
        }

        /**
         * Sets the id.
         *
         * @param id the new id
         */
        public void setId(final String id) {
            this.id = id;
        }

        /**
         * Gets the url.
         *
         * @return the url
         */
        public String getUrl() {
            return url;
        }

        /**
         * Sets the url.
         *
         * @param url the new url
         */
        public void setUrl(final String url) {
            this.url = url;
        }

        /**
         * Checks if is disable ssl validation.
         *
         * @return true, if is disable ssl validation
         */
        public boolean isDisableSslValidation() {
            return disableSslValidation;
        }

        /**
         * Sets the disable ssl validation.
         *
         * @param disableSslValidation the new disable ssl validation
         */
        public void setDisableSslValidation(final boolean disableSslValidation) {
            this.disableSslValidation = disableSslValidation;
        }

        /* (non-Javadoc)
         * @see java.lang.Object#equals(java.lang.Object)
         */
        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            WebHook webHook = (WebHook) o;

            if (disableSslValidation != webHook.disableSslValidation) {
                return false;
            }
            if (id != null ? !id.equals(webHook.id) : webHook.id != null) {
                return false;
            }
            return url != null ? url.equals(webHook.url) : webHook.url == null;

        }

        /* (non-Javadoc)
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            int result = id != null ? id.hashCode() : 0;
            final int prime = 31;
            result = prime * result + (url != null ? url.hashCode() : 0);
            result = prime * result + (disableSslValidation ? 1 : 0);
            return result;
        }

        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "WebHook{"
                    + "id='" + id + '\''
                    + ", url='" + url + '\''
                    + ", disableSslValidation=" + disableSslValidation
                    + '}';
        }
    }
}
