package com.perforce.hws.server;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.perforce.hws.p4base.AuthError;
import com.perforce.hws.p4base.P4BaseException;
import com.perforce.hws.p4base.ResultMap;
import com.perforce.hws.server.filters.*;
import com.perforce.hws.server.routes.*;
import com.perforce.hws.server.routes.git_fusion.*;
import com.perforce.hws.server.routes.perforce.*;
import com.perforce.p4java.Log;
import com.perforce.p4java.exception.MessageSeverityCode;
import com.perforce.p4java.server.callback.ILogCallback;
import org.apache.http.HttpStatus;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import spark.Service;
import spark.servlet.SparkApplication;

import javax.script.ScriptException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import static com.perforce.hws.server.ApiPath.*;
import static com.perforce.hws.server.ApiPath.Login;
import static com.perforce.hws.server.SetUIDMethods.*;


/**
 * Configures the Spark/Jetty web server and connects up all our routers and
 * middleware.<p/>
 */
public class HelixWebServices implements UrlBuilderMethods, StatusChecks,
        SparkApplication, ApiPathUtils {

    /**
     * The Constant logger.
     */
    private static final Logger LOGGER =
            LoggerFactory.getLogger(HelixWebServices.class);

    /**
     * The hws settings.
     */
    private HWSSettings hwsSettings = new HWSSettings();

    /**
     * The gson instance for converting json to java and
     * vice-versa.
     */
    private Gson gson;

    /**
     * The http request factory filter.
     */
    private HttpRequestFactoryFilter httpRequestFactoryFilter =
            new HttpRequestFactoryFilter();

    /**
     * By default, create a single HelixWebServices instance.
     *
     * @param args the arguments
     * @throws Exception the exception
     */
    public static void main(final String[] args) throws Exception {
        try {
            initializeLogging();
            HelixWebServices helixWebServices = new HelixWebServices();
            helixWebServices.init();
            LOGGER.info("Helix Web Services started");
        } catch (Exception e) {
            LOGGER.error("Shutting down system due to exception", e);
            throw e;
        }
    }

    /**
     * This used to initialize the application while running inside of another web
     * server.
     */
    @Override
    public void init() {
        try {
            Service service = Service.ignite();

            initializeServiceFromSettings(service);
            initializeStaticFileLocation(service);

            addBeforeFilters(service);
            addRoutes(service);
            addAfterFilters(service);
            addExceptionHandlers(service);

            setUser(service);

            // Log any status errors on startup.
            isStatusOk(hwsSettings);
        } catch (IOException io) {
            throw new UncheckedIOException(io);
        } catch (ScriptException | InterruptedException | ExecutionException se) {
            throw new IllegalStateException(se);
        }
    }

    /**
     * Generally just reads configuration from external
     * sources and preps the instance as well
     * as we know.
     *
     * If you haven't set up environent or system properties,
     * don't worry, just edit the HWSSettings
     * via the getSettings() method and associate it to
     * whatever configuration system you use.
     */
    public HelixWebServices() {
        gson = new GsonBuilder().
                setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ").create();

        LOGGER.info("initializing HelixWebServices");
        initializeSettings();
        LOGGER.info("HelixWebServices initialized");
    }

    /**
     * Initialize logging.
     *
     * @throws FileNotFoundException the file not found exception
     */
    private static void initializeLogging() throws FileNotFoundException {
        if (System.getProperty("log4j.configurationFile") == null) {
            LoggerContext context = (LoggerContext) LogManager.getContext(false);
            Configuration configuration = context.getConfiguration();
            configuration.getRootLogger().setLevel(Level.ALL);
            context.updateLoggers(configuration);
            LOGGER.info("No configuration file found, enabling all logging");
        }

        Log.setLogCallback(new ILogCallback() {
            @Override
            public void internalError(final String errorString) {
                LOGGER.error(errorString);
            }

            @Override
            public void internalException(final Throwable thr) {
                LOGGER.warn(thr.toString(), thr);
            }

            @Override
            public void internalWarn(final String warnString) {
                LOGGER.warn(warnString);
            }

            @Override
            public void internalInfo(final String infoString) {
                // P4Java gets pretty chatty.
                LOGGER.debug(infoString);
            }

            @Override
            public void internalStats(final String statsString) {
                LOGGER.debug(statsString);
            }

            @Override
            public void internalTrace(final LogTraceLevel traceLevel,
                                      final String traceMessage) {
                LOGGER.trace(traceMessage);
            }

            @Override
            public LogTraceLevel getTraceLevel() {
                if (LOGGER.isTraceEnabled()) {
                    return LogTraceLevel.ALL;
                } else {
                    return LogTraceLevel.COARSE;
                }
            }
        });
    }

    /**
     * Load settings and apply overrides from both the environment
     * and the configuration file.
     */
    private void initializeSettings() {
        try {
            hwsSettings.overrideFromEnvironment();

            String path = System.getenv("HWS_CONFIG_PATH");
            if (path != null) {
                hwsSettings.setSystemConfigPath(path);
            }
            path = System.getProperty("HWS_CONFIG_PATH");
            if (path != null) {
                hwsSettings.setSystemConfigPath(path);
            }
            hwsSettings.overrideFromSystemConfig();

            // Double check that the settings has at least one p4d
            // server configured, or throw a warning
            if (hwsSettings.getP4dConfigIds().isEmpty()) {
                LOGGER.warn(
                    "No p4d configurations found, you likely have misconfigured a "
                  + "P4DCONFIGDIR, or, you do not have any p4d configuration files");
            }
        } catch (IOException io) {
            throw new UncheckedIOException(io);
        }
    }

    /**
     * Some settings, like "is this HTTPS" or what port to listen to,
     * can be initialized from the HWS settings.
     *
     * If you don't want to use these settings, then don't call this method.
     *
     * @param service the service
     */
    public void initializeServiceFromSettings(final Service service) {
        if (hwsSettings.isEnableHttps()) {
            LOGGER.info("setting secure server: keystore {}, truststore {}",
                    hwsSettings.getKeystoreFile(),
                    hwsSettings.getTruststoreFile());
            service.secure(hwsSettings.getKeystoreFile(),
                    hwsSettings.getKeystorePassword(),
                    hwsSettings.getTruststoreFile(),
                    hwsSettings.getTruststorePassword());
        }

        service.port(hwsSettings.getHwsPort());
    }

    /**
     * Configure the service's static file location,
     * which basically points out where we've stored online documentation.
     *
     * @param service the service
     */
    public void initializeStaticFileLocation(final Service service) {
        service.staticFileLocation("/publicsite");
    }

    /**
     * Adds the before filters.
     *
     * @param service the service
     * @throws IOException     Signals that an I/O exception has occurred.
     * @throws ScriptException the script exception
     */
    public void addBeforeFilters(final Service service)
            throws IOException, ScriptException {
        service.before(new VersionFilter());
        service.before(new SettingsFilter(hwsSettings));
        service.before(new PlatformVersionFilter(hwsSettings));

        if (hwsSettings.getRequestFilterPath() != null) {
            service.before(new CustomRequestFilter(hwsSettings));
        }
        service.before(httpRequestFactoryFilter);
    }

    /**
     * Adds the default routes.
     *
     * @param service the service
     */
    private void addDefaultRoutes(final Service service) {
        //--------------------------------------------------------------------
        // Methods that really don't belong to a platform version
        //--------------------------------------------------------------------
        service.get("/", (request, response) -> {
            response.redirect(apiPathTo(Hws, Version));
            return null;
        });
        // This should remain a simple page that people who are lost can find
        // their way to something reasonable.
        service.get(apiPathTo(Hws, Version), new VersionRoute(hwsSettings));
        service.get(apiPathTo(P4d, Version), new PerforceVersionRoute(hwsSettings));
    }

    /**
     * Adds the hws routes.
     *
     * @param service the service
     */
    private void addHwsRoutes(final Service service) {
        //--------------------------------------------------------------------
        // Generic methods not tied to a specific server
        //--------------------------------------------------------------------
        // These methods should all precede with a platform version.
        service.get(hwsPathTo(Status), new StatusRoute(), gson::toJson);
        service.post(hwsPathTo(Login), new LoginRoute());
        service.get(hwsPathTo(Servers), new ConfigP4dsRoute(), gson::toJson);
    }

    /**
     * Adds the perforce routes.
     *
     * @param service the service
     */
    private void addPerforceRoutes(final Service service) {
        //--------------------------------------------------------------------
        // Most of these are "p4d commands in a method form"
        //--------------------------------------------------------------------
        service.get(serverPathTo(Branches),
                AuthenticatedRoute.create(new PerforceListBranchesRoute()),
                gson::toJson);
        service.post(serverPathTo(Branches),
                AuthenticatedRoute.create(new PerforceCreateBranchRoute()),
                gson::toJson);
        service.get(serverPathTo(Branches, Branch),
                AuthenticatedRoute.create(new PerforceGetBranchRoute()),
                gson::toJson);
        service.patch(serverPathTo(Branches, Branch),
                AuthenticatedRoute.create(new PerforceUpdateBranchRoute()),
                gson::toJson);
        service.delete(serverPathTo(Branches, Branch),
                AuthenticatedRoute.create(new PerforceDeleteBranchRoute()),
                gson::toJson);
        service.get(serverPathTo(Changes),
                AuthenticatedRoute.create(new PerforceListChangesRoute()),
                gson::toJson);
        service.post(serverPathTo(Changes),
                AuthenticatedRoute.create(new PerforceCreateChangeRoute()),
                gson::toJson);
        service.get(serverPathTo(Changes, Change),
                AuthenticatedRoute.create(new PerforceGetChangeRoute()),
                gson::toJson);
        service.get(serverPathTo(Clients),
                AuthenticatedRoute.create(new PerforceListClientsRoute()),
                gson::toJson);
        service.post(serverPathTo(Clients),
                AuthenticatedRoute.create(new PerforceCreateClientRoute()),
                gson::toJson);
        service.get(serverPathTo(Clients, Client),
                AuthenticatedRoute.create(new PerforceGetClientRoute()),
                gson::toJson);
        service.patch(serverPathTo(Clients, Client),
                AuthenticatedRoute.create(new PerforceUpdateClientRoute()),
                gson::toJson);
        service.delete(serverPathTo(Clients, Client),
                AuthenticatedRoute.create(new PerforceDeleteClientRoute()),
                gson::toJson);
        service.get(serverPathTo(Commands, Command),
                AuthenticatedRoute.create(new PerforceExecCommandRoute()),
                gson::toJson);
        service.post(serverPathTo(Commands, Command),
                AuthenticatedRoute.create(new PerforceExecInputCommandRoute()),
                gson::toJson);
        service.get(serverPathTo(Counters),
                AuthenticatedRoute.create(new PerforceListCountersRoute()),
                gson::toJson);
        service.get(serverPathTo(Counters, Counter),
                AuthenticatedRoute.create(new PerforceGetCounterRoute()),
                gson::toJson);
        service.put(serverPathTo(Counters, Counter),
                AuthenticatedRoute.create(new PerforceUpdateCounterRoute()),
                gson::toJson);
        service.post(serverPathTo(Counters, Counter, Increment),
                AuthenticatedRoute.create(new PerforceIncrementCounterRoute()),
                gson::toJson);
        service.delete(serverPathTo(Counters, Counter),
                AuthenticatedRoute.create(new PerforceDeleteCounterRoute()),
                gson::toJson);
        service.get(serverPathTo(Depots),
                AuthenticatedRoute.create(new PerforceListDepotsRoute()),
                gson::toJson);
        service.post(serverPathTo(Depots),
                AuthenticatedRoute.create(new PerforceCreateDepotRoute()),
                gson::toJson);
        service.get(serverPathTo(Depots, Depot),
                AuthenticatedRoute.create(new PerforceGetDepotRoute()),
                gson::toJson);
        service.patch(serverPathTo(Depots, Depot),
                AuthenticatedRoute.create(new PerforceUpdateDepotRoute()),
                gson::toJson);
        service.delete(serverPathTo(Depots, Depot),
                AuthenticatedRoute.create(new PerforceDeleteDepotRoute()),
                gson::toJson);
        service.get(serverPathTo(Groups),
                AuthenticatedRoute.create(new PerforceListGroupsRoute()),
                gson::toJson);
        service.post(serverPathTo(Groups),
                AuthenticatedRoute.create(new PerforceCreateGroupRoute()),
                gson::toJson);
        service.get(serverPathTo(Groups, Group),
                AuthenticatedRoute.create(new PerforceGetGroupRoute()),
                gson::toJson);
        service.patch(serverPathTo(Groups, Group),
                AuthenticatedRoute.create(new PerforceUpdateGroupRoute()),
                gson::toJson);
        service.delete(serverPathTo(Groups, Group),
                AuthenticatedRoute.create(new PerforceDeleteGroupRoute()),
                gson::toJson);
        service.get(serverPathTo(Jobs),
                AuthenticatedRoute.create(new PerforceListJobsRoute()),
                gson::toJson);
        service.post(serverPathTo(Jobs),
                AuthenticatedRoute.create(new PerforceCreateJobRoute()),
                gson::toJson);
        service.get(serverPathTo(Jobs, Job),
                AuthenticatedRoute.create(new PerforceGetJobRoute()),
                gson::toJson);
        service.patch(serverPathTo(Jobs, Job),
                AuthenticatedRoute.create(new PerforceUpdateJobRoute()),
                gson::toJson);
        service.delete(serverPathTo(Jobs, Job),
                AuthenticatedRoute.create(new PerforceDeleteJobRoute()),
                gson::toJson);
        service.post(serverPathTo(Jobs, Job, Fixes, Change),
                AuthenticatedRoute.create(new PerforceFixJobRoute()),
                gson::toJson);
        service.delete(serverPathTo(Jobs, Job, Fixes, Change),
                AuthenticatedRoute.create(new PerforceDeleteJobFixRoute()),
                gson::toJson);
        service.get(serverPathTo(Labels),
                AuthenticatedRoute.create(new PerforceListLabelsRoute()),
                gson::toJson);
        service.post(serverPathTo(Labels),
                AuthenticatedRoute.create(new PerforceCreateLabelRoute()),
                gson::toJson);
        service.get(serverPathTo(Labels, Label),
                AuthenticatedRoute.create(new PerforceGetLabelRoute()),
                gson::toJson);
        service.patch(serverPathTo(Labels, Label),
                AuthenticatedRoute.create(new PerforceUpdateLabelRoute()),
                gson::toJson);
        service.delete(serverPathTo(Labels, Label),
                AuthenticatedRoute.create(new PerforceDeleteLabelRoute()),
                gson::toJson);
        service.get(serverPathTo(Paths),
                AuthenticatedRoute.create(new PerforceListPathsRoute()),
                gson::toJson);
        service.post(serverPathTo(Login), new LoginRoute());
        service.get(serverPathTo(Protections),
                AuthenticatedRoute.create(new PerforceListProtectionsRoute()),
                gson::toJson);
        service.put(serverPathTo(Protections),
                AuthenticatedRoute.create(new PerforceUpdateProtectionsRoute()),
                gson::toJson);
        service.get(serverPathTo(Streams),
                AuthenticatedRoute.create(new PerforceListStreamsRoute()),
                gson::toJson);
        service.post(serverPathTo(Streams),
                AuthenticatedRoute.create(new PerforceCreateStreamRoute()),
                gson::toJson);
        service.get(serverPathTo(Streams, Stream),
                AuthenticatedRoute.create(new PerforceGetStreamRoute()),
                gson::toJson);
        service.patch(serverPathTo(Streams, Stream),
                AuthenticatedRoute.create(new PerforceUpdateStreamRoute()),
                gson::toJson);
        service.delete(serverPathTo(Streams, Stream),
                AuthenticatedRoute.create(new PerforceDeleteStreamRoute()),
                gson::toJson);
        service.get(serverPathTo(Triggers),
                AuthenticatedRoute.create(new PerforceListTriggersRoute()),
                gson::toJson);
        service.put(serverPathTo(Triggers),
                AuthenticatedRoute.create(new PerforceUpdateTriggersRoute()),
                gson::toJson);
        service.get(serverPathTo(Users),
                AuthenticatedRoute.create(new PerforceListUsersRoute()),
                gson::toJson);
        service.post(serverPathTo(Users),
                AuthenticatedRoute.create(new PerforceCreateUserRoute()),
                gson::toJson);
        service.get(serverPathTo(Users, User),
                AuthenticatedRoute.create(new PerforceGetUserRoute()),
                gson::toJson);
        service.patch(serverPathTo(Users, User),
                AuthenticatedRoute.create(new PerforceUpdateUserRoute()),
                gson::toJson);
        service.delete(serverPathTo(Users, User),
                AuthenticatedRoute.create(new PerforceDeleteUserRoute()),
                gson::toJson);
        service.get(serverPathTo(Servers),
                AuthenticatedRoute.create(new PerforceListServersRoute()),
                gson::toJson);
        service.post(serverPathTo(Servers),
                AuthenticatedRoute.create(new PerforceCreateServerRoute()),
                gson::toJson);
        service.get(serverPathTo(Servers, ServerId),
                AuthenticatedRoute.create(new PerforceGetServerRoute()),
                gson::toJson);
        service.patch(serverPathTo(Servers, ServerId),
                AuthenticatedRoute.create(new PerforceUpdateServerRoute()),
                gson::toJson);
        service.delete(serverPathTo(Servers, ServerId),
                AuthenticatedRoute.create(new PerforceDeleteServerRoute()),
                gson::toJson);
    }

    /**
     * Adds the git fusion routes.
     *
     * @param service the service
     */
    private void addGitFusionRoutes(final Service service) {
        //--------------------------------------------------------------------
        // Git Fusion related methods
        //--------------------------------------------------------------------
        if (hwsSettings.isEnableGitFusion()) {
            service.get(serverPathTo(GitFusionRepos),
                    AuthenticatedRoute.create(new GitFusionListRepoConfigsRoute()),
                    gson::toJson);
            service.post(serverPathTo(GitFusionRepos),
                    AuthenticatedRoute.create(new GitFusionCreateRepoConfigRoute()),
                    gson::toJson);
            service.get(serverPathTo(GitFusionRepos, Repo),
                    AuthenticatedRoute.create(new GitFusionGetRepoConfigRoute()),
                    gson::toJson);
            service.patch(serverPathTo(GitFusionRepos, Repo),
                    AuthenticatedRoute.create(new GitFusionUpdateRepoConfigRoute()),
                    gson::toJson);
            service.delete(serverPathTo(GitFusionRepos, Repo),
                    AuthenticatedRoute.create(new GitFusionDeleteRepoConfigRoute()),
                    gson::toJson);
        }
    }

    /**
     * Adds routes for the configured contexts. Currently known contexts
     * are:
     *
     * - Basic api routes
     * - Internal HWS routes
     * - Perforce Server routes
     * - Git fusion adapter routes
     *
     * @param service the service
     */
    public void addRoutes(final Service service) {
        addDefaultRoutes(service);
        addHwsRoutes(service);
        addGitFusionRoutes(service);
        addPerforceRoutes(service);
    }

    /**
     * Adds filters applied to the response after the main processing
     * is complete.
     *
     * Current filters are:
     *
     * - CORS filters, for resource sharing
     * - Content type of the response body
     *
     * @param service the service
     */
    public void addAfterFilters(final Service service) {
        service.after(new EnableCORSFilter(hwsSettings));
        service.after(new JsonContentTypeFilter());
    }

    /**
     * Adds the exception handlers.
     *
     * Conditions handled are:
     *
     * - Authentication error
     * - Unknown perforce server
     * - Unsupported encoding
     * - Other exception, reported as server error
     *
     * @param service the service
     */
    public void addExceptionHandlers(final Service service) {
        // Any exception handler you add here should call cleanUpOrLog.

        service.exception(AuthError.class, (e, request, response) -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Authentication Failed", e);
            }
            response.status(HttpStatus.SC_FORBIDDEN);
            response.body("");
        });

        service.exception(P4BaseException.class, (p4e, request, response) -> {
            LOGGER.error("Perforce exception", p4e);
            ResultMap p4Results = ((P4BaseException) p4e).getResultMap();
            List<Map<String, String>> responses =
                    p4Results.getErrors().stream().map(err -> {
                        Map<String, String> m = new HashMap<>();
                        m.put("MessageCode", err.getUniqueCode());
                        m.put("MessageText", p4e.getLocalizedMessage());
                        switch (err.getSeverity()) {
                            default:
                            case MessageSeverityCode.E_EMPTY:
                                m.put("MessageSeverity", "EMPTY");
                                break;

                            case MessageSeverityCode.E_INFO:
                                m.put("MessageSeverity", "INFO");
                                break;

                            case MessageSeverityCode.E_WARN:
                                m.put("MessageSeverity", "WARN");
                                break;

                            case MessageSeverityCode.E_FAILED:
                                m.put("MessageSeverity", "ERROR");
                                break;

                            case MessageSeverityCode.E_FATAL:
                                m.put("MessageSeverity", "FATAL");
                                break;
                        }
                        return m;
                    }).collect(Collectors.toList());

            // Our after filter may not get hit, so force json
            response.header("Content-Type", "application/json");
            // Set return status for the perforce command, improve maybe?
            switch (p4Results.getMaxSeverity()) {
                case MessageSeverityCode.E_FAILED:
                    response.status(HttpStatus.SC_BAD_REQUEST);
                    break;
                default:
                    response.status(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                    break;
            }
            // Set the body
            if (responses.size() > 1) {
                response.body(gson.toJson(responses));
            } else {
                response.body(gson.toJson(responses.get(0)));
            }
        });

        service.exception(UnknownServerException.class, (use, request, response) -> {
            String serverId = ((UnknownServerException) use).getServerId();
            LOGGER.error("Unknown server " + serverId, use);

            // Our after filter may not get hit.
            response.header("Content-Type", "application/json");
            // Set up the response for this error
            response.status(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            Map<String, String> map = new HashMap<>();
            map.put("MessageCode", "0");
            map.put("MessageSeverity", "ERROR");
            map.put("MessageText", "Unknown server ID [" + serverId + "]");
            response.body(gson.toJson(map));
        });

        service.exception(UnsupportedCharsetException.class,
                (uce, request, response) -> {

                    // A charset exception shouldn't be caused by any kind
                	// of user input.
                    Map<String, String> map = new HashMap<>();
                    map.put("MessageCode", "0");
                    map.put("MessageSeverity", "FATAL");
                    map.put("MessageText", "Server has a misconfigured charset");

                    // Our after filter may not get hit.
                    response.header("Content-Type", "application/json");
                    // Set up the response for this error
                    response.status(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                    response.body(gson.toJson(map));
                });

        service.exception(Exception.class, (e, request, response) -> {
            LOGGER.error("Unhandled exception", e);

            // Our after filter may not get hit.
            response.header("Content-Type", "application/json");
            response.status(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            response.body("");
        });
    }

    /**
     * As the system is configured for routes, etc, the Jetty server is launched
     * on a background thread. Once it's been launched, we want to allow the
     * process to change user accounts.
     *
     * @param service the new user
     * @throws ExecutionException   the execution exception
     * @throws InterruptedException the interrupted exception
     * @throws IOException          Signals that an I/O exception has occurred.
     */
    private void setUser(final Service service)
            throws ExecutionException, InterruptedException, IOException {

        if (hwsSettings.getSetuid() == 0) {
            return;
        }

        int currentUid = getuid();

        if (currentUid != hwsSettings.getSetuid() && currentUid != 0) {
            LOGGER.warn("Current UID is non-zero: {}, skipping setuid", getuid());
            return;
        }

        service.awaitInitialization();

        if (hwsSettings.getSetumask() != 0) {
            LOGGER.info("Setting umask to {}", hwsSettings.getSetumask());
            setumask(hwsSettings.getSetumask());
        }

        if (hwsSettings.getSetgid() != 0) {
            LOGGER.info("Setting gid to {}", hwsSettings.getSetgid());
            setgid(hwsSettings.getSetgid());
        }

        LOGGER.info("Setting uid to {}", hwsSettings.getSetuid());
        setuid(hwsSettings.getSetuid());
    }

    /**
     * Gets the json to/from java object converter.
     *
     * @return the gson
     */
    public Gson getGson() {
        return gson;
    }

    /* (non-Javadoc)
     * @see com.perforce.hws.server.ApiPathUtils#getSettings()
     */
    @Override
    public HWSSettings getSettings() {
        return hwsSettings;
    }
}
