package com.perforce.hws.server;

import com.perforce.hws.p4base.AuthMethods;
import com.perforce.hws.p4base.ServerHandle;
import com.perforce.hwsclient.models.LoginRequest;
import com.perforce.hwsclient.models.LoginResponse;
import com.perforce.p4java.exception.AccessException;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.Date;

/**
 * Provides methods to create sessions via logging in.
 */
public interface Login extends UsesServerHandles, AuthMethods {

    /** The logger. */
    Logger LOGGER = LoggerFactory.getLogger(Login.class);

    /**
     * Will attempt to login to the the specified p4d which should
     * have be determined from the request parameters or default
     * authorization settings. If login is successful a ticket 
     * will be returned. This ticket will be in the form of 
     * user:p4ticket (base64 encoded) signed with the signing key.
     *
     * @param loginRequest the login request
     * @param settings the settings
     * @param p4dServerId the p4d server id
     * @return null if we are unable to log in to the specified server.
     */
    default LoginResponse login(LoginRequest loginRequest,
                                HWSSettings settings,
                                String p4dServerId) {
        String p4ticket = null;
        try (ServerHandle serverHandle = obtainServerHandle(p4dServerId, settings)) {
            if (serverHandle != null) {
                String user = loginRequest.getUser();
                String password = loginRequest.getPassword();
                p4ticket = logIn(serverHandle, user, password, true);
            }
        } catch (AccessException ex) {
            LOGGER.debug("p4d login failed to server " + p4dServerId, ex);
        }

        if (p4ticket == null) {
            return null;
        }

        String credentialsString = loginRequest.getUser() + ":" + p4ticket;
        String encodedString =
        		new String(Base64.getEncoder().encode(credentialsString.getBytes()),
                Charset.forName("UTF-8"));

        JwtBuilder jwtBuilder = Jwts.builder().setId(encodedString);
        jwtBuilder.setExpiration(Date.from(Instant.now().plus(
        		settings.getJwtTimeoutInSeconds(), ChronoUnit.SECONDS)));
        jwtBuilder.signWith(SignatureAlgorithm.HS512, settings.getJwtSigningKey());
        String ticket = jwtBuilder.compact();
        LoginResponse loginResponse = new LoginResponse();
        loginResponse.setTicket(ticket);
        return loginResponse;
    }
}
