package com.perforce.hws.server;

import com.perforce.hws.p4base.P4Methods;
import com.perforce.hws.p4base.P4ResultsException;
import com.perforce.hws.p4base.ResultMap;
import com.perforce.hws.p4base.ServerHandle;
import com.perforce.hws.util.MapUtils;
import com.perforce.hwsclient.models.*;

import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * Methods to help convert between the p4java ResultMap and the client SDK
 * models.
 */
public interface ModelConversion extends P4Methods, MapUtils {

    /**
     * Convert our results to a CommandResponse
     *
     * @param results
     * @return
     */
    default CommandResponse toCommandResponse(List<? extends Map<String, Object>> results) {
        CommandResponse commandResponse = new CommandResponse();
        commandResponse.setResults((List<Map<String, Object>>) results);
        return commandResponse;
    }

    default BranchCommand toBranchCommand(ResultMap resultMap, Supplier<String> tzoffset) {
        BranchCommand branchCommand = new BranchCommand();
        setIfExists(branchCommand::setBranch, "Branch", resultMap);
        setIfExists(branchCommand::setOwner, "Owner", resultMap);
        setDateIfExists(branchCommand::setAccess, "Access", resultMap, tzoffset);
        setDateIfExists(branchCommand::setUpdate, "Update", resultMap, tzoffset);
        setIfExists(branchCommand::setOptions, "Options", resultMap);
        setIfExists(branchCommand::setDescription, "Description", resultMap);
        setCollated(branchCommand::setView, "View", resultMap);

        return branchCommand;
    }

    default BranchesCommand toBranchesCommand(ResultMap resultMap, Supplier<String> tzoffset) {
        BranchesCommand branchesCommand = new BranchesCommand();

        setIfExists(branchesCommand::setBranch, "branch", resultMap);
        setIfExists(branchesCommand::setOwner, "Owner", resultMap);
        setDateIfExists(branchesCommand::setAccess, "Access", resultMap, tzoffset);
        setDateIfExists(branchesCommand::setUpdate, "Update", resultMap, tzoffset);
        setIfExists(branchesCommand::setOptions, "Options", resultMap);
        setIfExists(branchesCommand::setDescription, "Description", resultMap);

        return branchesCommand;
    }

    default ChangeCommand toChangeCommand(ResultMap map,
                                          Supplier<String> offsetSupplier) {
        ChangeCommand changeCommand = new ChangeCommand();

        setIfExists(changeCommand::setChange, "Change", map);
        setIfExists(changeCommand::setClient, "Client", map);
        setDateIfExists(changeCommand::setDate, "Date", map, offsetSupplier);
        setIfExists(changeCommand::setUser, "User", map);
        setIfExists(changeCommand::setStatus, "Status", map);
        setIfExists(changeCommand::setDescription, "Description", map);
        setCollated(changeCommand::setJobs, "Jobs", map);
        setIfExists(changeCommand::setType, "Type", map);
        setCollated(changeCommand::setFiles, "Files", map);
        setIfExists(changeCommand::setImportedBy, "ImportedBy", map);
        setIfExists(changeCommand::setIdentify, "Identity", map);

        return changeCommand;
    }

    default ChangesCommand toChangesCommand(ResultMap resultMap,
                                            Supplier<String> offsetSupplier) {
        ChangesCommand changesCommand = new ChangesCommand();

        setIfExists(changesCommand::setChange, "change", resultMap);
        setDateIfExists(changesCommand::setDate, "time", resultMap, offsetSupplier);
        setIfExists(changesCommand::setUser, "user", resultMap);
        setIfExists(changesCommand::setClient, "client", resultMap);
        setIfExists(changesCommand::setStatus, "status", resultMap);
        setIfExists(changesCommand::setType, "changeType", resultMap);
        setIfExists(changesCommand::setPath, "path", resultMap);
        setIfExists(changesCommand::setDescription, "desc", resultMap);

        return changesCommand;
    }

    default ClientCommand toClientCommand(ResultMap map,
                                          Supplier<String> offsetSupplier) {
        ClientCommand command = new ClientCommand();

        setIfExists(command::setClient, "Client", map);
        setIfExists(command::setOwner, "Owner", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setHost, "Host", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setRoot, "Root", map);
        setCollated(command::setAltRoots, "AltRoots", map);
        setIfExists(command::setOptions, "Options", map);
        setIfExists(command::setSubmitOptions, "SubmitOptions", map);
        setIfExists(command::setLineEnd, "LineEnd", map);
        setIfExists(command::setStream, "Stream", map);
        setIfExists(command::setStreamAtChange, "StreamAtChange", map);
        setIfExists(command::setServerID, "ServerID", map);
        setCollated(command::setView, "View", map);
        setCollated(command::setChangeView, "ChangeView", map);
        setIfExists(command::setType, "Type", map);

        return command;
    }

    default ClientsCommand toClientsCommand(ResultMap map,
                                            Supplier<String> offsetSupplier) {
        ClientsCommand command = new ClientsCommand();

        setIfExists(command::setClient, "client", map);
        setIfExists(command::setOwner, "Owner", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setHost, "Host", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setRoot, "Root", map);
        setIfExists(command::setOptions, "Options", map);
        setIfExists(command::setSubmitOptions, "SubmitOptions", map);
        setIfExists(command::setLineEnd, "LineEnd", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setStream, "Stream", map);

        return command;
    }

    default Counter toCounter(ResultMap map) {
        Counter counter = new Counter();

        setIfExists(counter::setCounter, "counter", map);
        setIfExists(counter::setValue, "value", map);

        return counter;
    }

    default DepotCommand toDepotCommand(ResultMap map) {
        DepotCommand command = new DepotCommand();

        setIfExists(command::setDepot, "Depot", map);
        setIfExists(command::setOwner, "Owner", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setAddress, "Address", map);
        setIfExists(command::setSuffix, "Suffix", map);
        setIfExists(command::setStreamDepth, "StreamDepth", map);
        setIfExists(command::setMap, "Map", map);
        setCollated(command::setSpecMap, "SpecMap", map);

        return command;
    }

    default DepotsCommand toDepotsCommand(ResultMap map) {
        DepotsCommand command = new DepotsCommand();

        setIfExists(command::setDepot, "name", map);
        setIfExists(command::setType, "type", map);
        setIfExists(command::setStreamDepth, "depth", map);
        setIfExists(command::setMap, "map", map);
        setIfExists(command::setDescription, "desc", map);

        return command;
    }

    default DirsCommand toDirsCommand(ResultMap map) {
        DirsCommand command = new DirsCommand();

        setIfExists(command::setDir, "dir", map);

        return command;
    }

    default FilesCommand toFilesCommand(ResultMap map, Supplier<String> offsetSupplier) {
        FilesCommand command = new FilesCommand();

        setIfExists(command::setDepotFile, "depotFile", map);
        setIfExists(command::setRevision, "rev", map);
        setIfExists(command::setChange, "change", map);
        setIfExists(command::setAction, "action", map);
        setIfExists(command::setType, "type", map);
        setDateIfExists(command::setTime, "time", map, offsetSupplier);

        return command;
    }

    default FstatCommand toFstatCommand(ResultMap resultMap, Supplier<String> offsetSupplier) {
        // We duplicate the map to allow us to safely remove keys that might cause problems in
        // conversion (fstat is an odd duck).
        ResultMap map = new ResultMap(resultMap);

        FstatCommand command = new FstatCommand();

        setIfExists(command::setDepotFile, "depotFile", map);
        setIfExists(command::setMovedFile, "movedFile", map);
        setIfExists(command::setShelved, "shelved", map);
        setIfExists(command::setHeadAction, "headAction", map);
        setIfExists(command::setHeadChange, "headChange", map);
        setIfExists(command::setHeadRev, "headRev", map);
        setIfExists(command::setHeadType, "headType", map);
        setIfExists(command::setHeadCharset, "headCharset", map);
        setDateIfExists(command::setHeadTime, "headTime", map, offsetSupplier);
        setDateIfExists(command::setHeadModTime, "headModTime", map, offsetSupplier);
        setIfExists(command::setMovedRev, "movedRev", map);
        setIfExists(command::setDigest, "digest", map);
        setIfExists(command::setFileSize, "fileSize", map);
        setIfExists(command::setActionOwner, "actionOwner", map);
        setIfExists(command::setResolved, "resolved", map);
        setIfExists(command::setUnresolved, "unresolved", map);
        setIfExists(command::setReresolvable, "reresolvable", map);

        Map<String, Consumer<List<String>>> arrays = new HashMap<>();
        arrays.put("otherOpen", command::setOtherOpens);
        arrays.put("otherLock", command::setOtherLocks);
        arrays.put("otherAction", command::setOtherActions);
        arrays.put("otherChange", command::setOtherChanges);
        arrays.put("resolveActions", command::setResolveActions);
        arrays.put("resolveBaseFile", command::setResolveBaseFiles);
        arrays.put("resolveBaseRevs", command::setResolveBaseRevs);
        arrays.put("resolveFromFiles", command::setResolveFromFiles);
        arrays.put("resolveStartFromRevs", command::setResolveStartFromRevs);
        arrays.put("resolveEndFromRevs", command::setResolveEndFromRevs);

        arrays.forEach((key,consumer) -> {
            // Some fstat commands have a normal "otherOpen" key that just lists the number of
            // follow-up keys. I want to ignore these keys in setCollated, which is why we make
            // a copy of the input result map.
            if (map.containsKey(key)) {
                map.remove(key);
            }
            setCollated(consumer, key, map);
        });

        return command;
    }

    default GroupCommand toGroupCommand(Map<String, Object> map) {
        GroupCommand command = new GroupCommand();

        setIfExists(command::setGroup, "Group", map);
        setIfExists(command::setMaxResults, "MaxResults", map);
        setIfExists(command::setMaxScanRows, "MaxScanRows", map);
        setIfExists(command::setMaxLockTime, "MaxLockTime", map);
        setIfExists(command::setMaxOpenFiles, "MaxOpenFiles", map);
        setIfExists(command::setTimeout, "Timeout", map);
        setIfExists(command::setPasswordTimeout, "PasswordTimeout", map);
        setIfExists(command::setLdapConfig, "LdapConfig", map);
        setIfExists(command::setLdapSearchQuery, "LdapSearchQuery", map);
        setIfExists(command::setLdapUserAttribute, "LdapUserAttribute", map);
        setCollated(command::setSubgroups, "Subgroups", map);
        setCollated(command::setOwners, "Owners", map);
        setCollated(command::setUsers, "Users", map);

        return command;
    }

    default GroupsCommand toGroupsCommand(ResultMap map) {
        GroupsCommand command = new GroupsCommand();

        setIfExists(command::setUser, "user", map);
        setIfExists(command::setGroup, "group", map);
        setIfExists(command::setIsSubGroup, "isSubGroup", map);
        setIfExists(command::setIsOwner, "isOwner", map);
        setIfExists(command::setIsUser, "isUser", map);
        setIfExists(command::setMaxResults, "maxResults", map);
        setIfExists(command::setMaxScanRows, "maxScanRows", map);
        setIfExists(command::setMaxOpenFiles, "maxOpenFiles", map);
        setIfExists(command::setMaxLockTime, "maxLockTime", map);
        setIfExists(command::setTimeout, "timeout", map);
        setIfExists(command::setPassTimeout, "passTimeout", map);

        return command;
    }

    default JobCommand toJobCommand(ResultMap map) {
        JobCommand command = new JobCommand();

        command.putAll(map);
        setIfExists(command::setJob, "Job", map);

        return command;
    }

    default JobsCommand toJobsCommand(ResultMap map) {
        JobsCommand command = new JobsCommand();

        command.putAll(map);
        setIfExists(command::setJob, "Job", map);

        return command;
    }

    default LabelCommand toLabelCommand(ResultMap map, Supplier<String> offsetSupplier) {
        LabelCommand command = new LabelCommand();

        setIfExists(command::setLabel, "Label", map);
        setIfExists(command::setOwner, "Owner", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setOptions, "Options", map);
        setIfExists(command::setRevision, "Revision", map);
        setCollated(command::setView, "View", map);
        setIfExists(command::setServerID, "ServerID", map);

        return command;
    }

    default LabelsCommand toLabelsCommand(ResultMap map, Supplier<String> offsetSupplier) {
        LabelsCommand command = new LabelsCommand();

        setIfExists(command::setLabel, "label", map);
        setIfExists(command::setOwner, "Owner", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setOptions, "Options", map);

        return command;
    }

    default Location toLocation(ResultMap map, Supplier<String> offsetSupplier) {
        Location path = new Location();

        if (map.containsKey("content")) {
            path.setContent((String)map.get("content"));
            path.setFstat(toFstatCommand(map, offsetSupplier));
            path.setDepotPath(path.getFstat().getDepotFile());
        } else if (map.containsKey("depotFile")) {
            path.setFile(toFilesCommand(map, offsetSupplier));
            path.setDepotPath(path.getFile().getDepotFile());
        } else if (map.containsKey("dir")) {
            path.setDir(toDirsCommand(map));
            path.setDepotPath(path.getDir().getDir());
        } else {
            path.setDepot(toDepotsCommand(map));
            path.setDepotPath("//" + path.getDepot().getDepot());
        }

        return path;
    }

    default Protections toProtections(ResultMap map) {
        Protections protections = new Protections();
        setCollated(protections::setProtections, "Protections", map);
        return protections;
    }

    default ServerCommand toServerCommand(ResultMap map) {
        ServerCommand command = new ServerCommand();

        setIfExists(command::setServerID, "ServerID", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setServices, "Services", map);
        setIfExists(command::setName, "Name", map);
        setIfExists(command::setAddress, "Address", map);
        setIfExists(command::setExternalAddress, "ExternalAddress", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setUser, "User", map);
        setIfExists(command::setClientDataFilter, "ClientDataFilter", map);
        setIfExists(command::setRevisionDataFilter, "RevisionDataFilter", map);
        setIfExists(command::setArchiveDataFilter, "ArchiveDataFilter", map);
        setIfExists(command::setDistributedConfig, "DistributedConfig", map);

        return command;
    }

    default ServersCommand toServersCommand(ResultMap map) {
        ServersCommand command = new ServersCommand();

        setIfExists(command::setServerID, "ServerID", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setServices, "Services", map);
        setIfExists(command::setName, "Name", map);
        setIfExists(command::setAddress, "Address", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setUser, "User", map);

        return command;
    }

    default StreamCommand toStreamCommand(ResultMap map, Supplier<String> offsetSupplier) {
        StreamCommand command = new StreamCommand();

        setIfExists(command::setStream, "Stream", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setOwner, "Owner", map);
        setIfExists(command::setName, "Name", map);
        setIfExists(command::setParent, "Parent", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setOptions, "Options", map);
        setCollated(command::setPaths, "Paths", map);
        setCollated(command::setRemapped, "Remapped", map);
        setCollated(command::setIgnored, "Ignored", map);

        return command;
    }

    default StreamsCommand toStreamsCommand(ResultMap map, Supplier<String> offsetSupplier) {
        StreamsCommand command = new StreamsCommand();

        setIfExists(command::setStream, "Stream", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setOwner, "Owner", map);
        setIfExists(command::setName, "Name", map);
        setIfExists(command::setParent, "Parent", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setDescription, "Description", map);
        setIfExists(command::setOptions, "Options", map);

        return command;
    }

    default Triggers toTriggers(ResultMap map) {
        Triggers triggers = new Triggers();

        setCollated(triggers::setTriggers, "Triggers", map);

        return triggers;
    }

    default UserCommand toUserCommand(ResultMap map,
                                      Supplier<String> offsetSupplier) {
        UserCommand command = new UserCommand();

        setIfExists(command::setUser, "User", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setAuthMethod, "AuthMethod", map);
        setIfExists(command::setEmail, "Email", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setFullName, "FullName", map);
        setIfExists(command::setJobView, "JobView", map);
        setIfExists(command::setPassword, "Password", map);
        setDateIfExists(command::setPasswordChange, "PasswordChange", map, offsetSupplier);
        setCollated(command::setReviews, "Reviews", map);

        return command;
    }

    default UsersCommand toUsersCommand(ResultMap map,
                                        Supplier<String> offsetSupplier) {
        UsersCommand command = new UsersCommand();

        setIfExists(command::setUser, "User", map);
        setIfExists(command::setType, "Type", map);
        setIfExists(command::setEmail, "Email", map);
        setDateIfExists(command::setUpdate, "Update", map, offsetSupplier);
        setDateIfExists(command::setAccess, "Access", map, offsetSupplier);
        setIfExists(command::setFullName, "FullName", map);
        setIfExists(command::setHasPassword, "Password", map);

        return command;
    }

    /**
     * Converts the BranchCommand to a map suitable for using on "-i" command
     * queries.
     *
     * @param branchCommand
     * @return
     */
    default Map<String, Object> toRequestMap(BranchCommand branchCommand) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Branch", branchCommand::getBranch);
        putIfSet(map, "Owner", branchCommand::getOwner);
        putIfSet(map, "Options", branchCommand::getOptions);
        putIfSet(map, "Description", branchCommand::getDescription);
        putDivided(map, "View", branchCommand::getView);

        return map;
    }

    default Map<String, Object> toRequestMap(ClientCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Client", command::getClient);
        putIfSet(map, "Owner", command::getOwner);
        putIfSet(map, "Host", command::getHost);
        putIfSet(map, "Description", command::getDescription);
        putIfSet(map, "Root", command::getRoot);
        putDivided(map, "AltRoots", command::getAltRoots);
        putIfSet(map, "Options", command::getOptions);
        putIfSet(map, "SubmitOptions", command::getSubmitOptions);
        putIfSet(map, "LineEnd", command::getLineEnd);
        putIfSet(map, "Stream", command::getStream);
        putIfSet(map, "StreamAtChange", command::getStreamAtChange);
        putIfSet(map, "ServerID", command::getServerID);
        putDivided(map, "View", command::getView);
        putDivided(map, "ChangeView", command::getChangeView);
        putIfSet(map, "Type", command::getType);

        return map;
    }

    default Map<String, Object> toRequestMap(DepotCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Depot", command::getDepot);
        putIfSet(map, "Owner", command::getOwner);
        putIfSet(map, "Description", command::getDescription);
        putIfSet(map, "Type", command::getType);
        putIfSet(map, "Address", command::getAddress);
        putIfSet(map, "Suffix", command::getSuffix);
        putIfSet(map, "StreamDepth", command::getStreamDepth);
        putIfSet(map, "Map", command::getMap);
        putDivided(map, "SpecMap", command::getSpecMap);

        return map;
    }

    default Map<String, Object> toRequestMap(GroupCommand command) {
        Map<String, Object> map = new HashMap<>();
        
        putIfSet(map, "Group", command::getGroup);
        putIfSet(map, "MaxResults", command::getMaxResults);
        putIfSet(map, "MaxScanRows", command::getMaxScanRows);
        putIfSet(map, "MaxLockTime", command::getMaxLockTime);
        putIfSet(map, "MaxOpenFiles", command::getMaxOpenFiles);
        putIfSet(map, "Timeout", command::getTimeout);
        putIfSet(map, "PasswordTimeout", command::getPasswordTimeout);
        putIfSet(map, "LdapConfig", command::getLdapConfig);
        putIfSet(map, "LdapSearchQuery", command::getLdapSearchQuery);
        putIfSet(map, "LdapUserAttribute", command::getLdapUserAttribute);
        putDivided(map, "Subgroups", command::getSubgroups);
        putDivided(map, "Owners", command::getOwners);
        putDivided(map, "Users", command::getUsers);

        return map;
    }

    default Map<String, Object> toRequestMap(LabelCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Label", command::getLabel);
        putIfSet(map, "Owner", command::getOwner);
        putIfSet(map, "Description", command::getDescription);
        putIfSet(map, "Options", command::getOptions);
        putIfSet(map, "Revision", command::getRevision);
        putDivided(map, "View", command::getView);
        putIfSet(map, "ServerID", command::getServerID);

        return map;
    }

    default Map<String, Object> toRequestMap(Protections protections) {
        Map<String, Object> map = new HashMap<>();

        putDivided(map, "Protections", protections::getProtections);

        return map;
    }

    default Map<String, Object> toRequestMap(ServerCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "ServerID", command::getServerID);
        putIfSet(map, "Type", command::getType);
        putIfSet(map, "Services", command::getServices);
        putIfSet(map, "Name", command::getName);
        putIfSet(map, "Address", command::getAddress);
        putIfSet(map, "ExternalAddress", command::getExternalAddress);
        putIfSet(map, "Description", command::getDescription);
        putIfSet(map, "User", command::getUser);
        putIfSet(map, "ClientDataFilter", command::getClientDataFilter);
        putIfSet(map, "RevisionDataFilter", command::getRevisionDataFilter);
        putIfSet(map, "ArchiveDataFilter", command::getArchiveDataFilter);
        putIfSet(map, "DistributedConfig", command::getDistributedConfig);
        
        return map;
    }

    default Map<String, Object> toRequestMap(StreamCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Stream", command::getStream);
        putIfSet(map, "Owner", command::getOwner);
        putIfSet(map, "Name", command::getName);
        putIfSet(map, "Parent", command::getParent);
        putIfSet(map, "Type", command::getType);
        putIfSet(map, "Description", command::getDescription);
        putIfSet(map, "Options", command::getOptions);
        putDivided(map, "Paths", command::getPaths);
        putDivided(map, "Remapped", command::getRemapped);
        putDivided(map, "Ignored", command::getIgnored);

        return map;
    }

    default Map<String, Object> toRequestMap(Triggers triggers) {
        Map<String, Object> map = new HashMap<>();

        putDivided(map, "Triggers", triggers::getTriggers);

        return map;
    }

    default Map<String, Object> toRequestMap(UserCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "User", command::getUser);
        putIfSet(map, "Type", command::getType);
        putIfSet(map, "AuthMethod", command::getAuthMethod);
        putIfSet(map, "Email", command::getEmail);
        putIfSet(map, "FullName", command::getFullName);
        putIfSet(map, "JobView", command::getJobView);
        putIfSet(map, "Password", command::getPassword);
        putDivided(map, "Reviews", command::getReviews);

        return map;
    }

    default Map<String, Object> toRequestMap(JobCommand command) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "Job", command::getJob);

        map.putAll(command);

        return map;
    }

    default Map<String, Object> toRequestMap(Counter counter) {
        Map<String, Object> map = new HashMap<>();

        putIfSet(map, "counter", counter::getCounter);
        putIfSet(map, "value", counter::getValue);

        return map;
    }

    /**
     * Converts the ResultMap value to a java.util.Date timestamp.
     *
     * @param consumer The method that accepts the Date value
     * @param key      The key that holds the Date field in map
     * @param map      The result map that probably has a numeric-style or localized date string
     * @param offset   The method to grab the offset value (to convert the time to UTC)
     */
    default void setDateIfExists(Consumer<Date> consumer,
                                 String key,
                                 ResultMap map,
                                 Supplier<String> offset) {
        if (map.containsKey(key)) {
            Object value = map.get(key);
            if (value instanceof String) {
                String string = (String) value;
                // Some values are "numeric" which are expected to be Unix epoch
                // values.
                if (string.matches("^\\d+$")) {
                    long epoch = Long.valueOf(string);
                    consumer.accept(Date.from(Instant.ofEpochSecond(epoch)));
                } else {
                    // We're probably dealing with a 'formatted' date.
                    // ANNOYING: We need to grab the timezone from the server
                    // 'offset' information value.
                    DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ssZZZZZ");
                    OffsetDateTime dateTime = OffsetDateTime.parse(string + offset.get(), formatter);
                    Date date = Date.from(dateTime.toInstant());
                    consumer.accept(date);
                }
            } else {
                throw new IllegalStateException("Don't know how to convert " +
                        value.getClass() + " to Date: " + value);
            }
        }
    }

    /**
     * Generates an "offset string" either based from the tzoffset field or
     * by parsing the offset in the serverDate.
     * <p>
     * Examples:
     * <p>
     * tzoffset   -> "-25002"
     * serverDate -> "2016/03/18 06:46:17 -0700 PDT"
     *
     * @param serverHandle
     * @return
     */
    default Supplier<String> offsetSupplier(ServerHandle serverHandle) {
        return () -> {
            List<ResultMap> results = exec(serverHandle, "info");
            ResultMap map = results.get(0);

            if (map.containsKey("tzoffset")) {
                int tzoffset = Integer.valueOf((String) map.get("tzoffset"));
                return ZoneOffset.ofTotalSeconds(tzoffset).getId();
            } else {
                // This was the observed value in 15.2
                String serverDate = (String) map.get("serverDate");
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss XXXX zzz");
                OffsetDateTime dateTime = OffsetDateTime.parse(serverDate, formatter);
                return dateTime.getOffset().getId();
            }
        };
    }


    /**
     * Many collection values in p4java land use a "prefixed key entry"
     * approach; this will separate a collection into those key entries for
     * submission into p4java.
     *
     * @param map      The map to update
     * @param prefix   The key prefix to use, we'll append an index entry.
     * @param supplier The collection to use (if null we do nothing)
     * @param <T>      The type of entry in the collection
     */
    default <T> void putDivided(Map<String, Object> map, String prefix, Supplier<List<T>> supplier) {
        List<T> list = supplier.get();
        if (list != null) {
            for (int index = 0; index < list.size(); index++) {
                String key = String.format("%s%d", prefix, index);
                map.put(key, list.get(index));
            }
        }
    }

    default <T> void setCollated(Consumer<List<T>> consumer, String prefix, Map<String, Object> map) {
        List<String> keys =
                map.keySet().stream()
                        .filter(key -> key.startsWith(prefix))
                        .sorted((m, n) -> {
                            int mm = Integer.valueOf(m.substring(prefix.length()));
                            int nn = Integer.valueOf(n.substring(prefix.length()));
                            return Integer.compare(mm, nn);
                        })
                        .collect(Collectors.toList());

        // Don't call the consumer if there's nothing to do
        if (keys.isEmpty()) {
            return;
        }

        List<T> values = keys.stream().map(k -> (T) map.get(k)).collect(Collectors.toList());

        consumer.accept(values);
    }

    default ResultMap firstResultOrFail(List<ResultMap> resultMaps) {
        if (resultMaps.size() != 1) {
            throw new P4ResultsException(
                    "Expected only one result map from server", resultMaps);
        }
        return resultMaps.get(0);
    }
}
