package com.perforce.hws.server;

import com.perforce.hws.p4base.P4Methods;
import com.perforce.hws.p4base.ResultMap;
import com.perforce.p4java.exception.P4JavaException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.UnsupportedCharsetException;
import java.util.List;

/**
 * Utilities to verify the running state of the server.
 */
public interface StatusChecks extends UsesServerHandles, P4Methods {

    Logger logger = LoggerFactory.getLogger(StatusChecks.class);

    /**
     * Returns the top-level monitoring state.
     *
     * <p>Any errors or warnings should be printed to the log.
     *
     * @return True if we can access all p4ds.
     */
    default boolean isStatusOk(HWSSettings settings) {
        return allP4dsOk(settings);
    }

    /**
     * Checks all p4ds configured in the system.
     *
     * @param settings
     * @return
     */
    default boolean allP4dsOk(HWSSettings settings) {
        return settings.getP4dConfigMap().entrySet()
                .stream()
                .allMatch(e -> p4dIsOk(e.getKey(), settings));
    }

    /**
     * Checks the info from the server with indicated ID, just making sure we can
     * connect to it.
     *
     * @param serverId
     * @param settings
     * @return
     */
    default Boolean p4dIsOk(String serverId,
                            HWSSettings settings) {
        Boolean ok;
        try {
            ok = withSessionlessServerHandle(serverId, settings, serverHandle -> {
                List<ResultMap> results = exec(serverHandle, "info");
                return results != null;
            });
        } catch (UnsupportedCharsetException charsetEx) {
            P4dConfig config = settings.getP4dConfigMap().get(serverId);
            String charset = config != null ? config.getP4CHARSET() : "";
            logger.error("Unsuppored charset {} for p4d config ID {}", charset, serverId);
            // We've logged an informative message, no need to log more
            return false;
        } catch (RuntimeException ex) {
            ok = false;
        }
        boolean isOk = Boolean.TRUE.equals(ok);
        if (isOk) {
            logger.debug("'p4 info' ok for p4d config id {}", serverId);
        } else {
            logger.error("'p4 info' failed on p4d config id {}", serverId);
        }
        return isOk;
    }
}
