package com.perforce.hws.server;

import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.UriTemplate;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.gson.GsonFactory;
import com.perforce.hws.p4base.HasGson;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by tjuricek on 12/17/15.
 */
public interface UrlBuilderMethods extends HasGson {

    default JsonFactory getJsonFactory() {
        return new GsonFactory();
    }

    /**
     * Creates a URL for our request framework based on a possibly templated
     * path.
     *
     * @param baseUrl    We use this URL to grab the scheme, host, port, and any
     *                   "prefix" path.
     * @param subPath    This is a path "template" that can be safely parameterized
     *                   by the pathParams parameter.
     * @param pathParams The data for the path template. Can be null.
     * @param uriParams  If not null, we'll append these to the URI
     * @return Our reconfigured URL
     */
    default GenericUrl buildUrl(String baseUrl,
                                String subPath,
                                Object pathParams,
                                Map<String, String> uriParams) {

        // We use the java.net classes in this manner to detect most illegal
        // format strings.
        URI url = null;
        try {
            url = new URL(baseUrl).toURI();
        } catch (URISyntaxException | MalformedURLException e) {
            throw new IllegalArgumentException(e);
        }

        StringBuilder sb = new StringBuilder("{scheme}://{host}");

        Map<String, Object> p = new HashMap<>();
        p.put("scheme", url.getScheme());
        p.put("host", url.getHost());
        if (0 < url.getPort()) {
            sb.append(":{port}");
            p.put("port", url.getPort());
        }
        if (url.getPath() != null && !url.getPath().isEmpty()) {
            sb.append("{prefix}");
            p.put("prefix", url.getPath());
        }
        if (subPath != null) {
            if (pathParams != null) {
                sb.append(UriTemplate.expand(subPath, pathParams, false));
            } else {
                sb.append(subPath);
            }
        }

        String noParams = UriTemplate.expand(sb.toString(), p, false);

        String expanded = noParams;

        if (uriParams != null) {
            expanded = UriTemplate.expand(noParams, uriParams, true);
        }

        return new GenericUrl(expanded);
    }

    /**
     * A variation of buildUrl where we allow the baseUrl to be configured with
     * various path parameters, and generally do not validate it.
     *
     * @param baseUrl
     * @param pathParams
     * @param uriParams
     * @return
     */
    default GenericUrl buildUrl(String baseUrl,
                                Object pathParams,
                                Map<String, String> uriParams) {

        String noParams = UriTemplate.expand(baseUrl, pathParams, false);

        String expanded = noParams;

        if (uriParams != null) {
            expanded = UriTemplate.expand(noParams, uriParams, true);
        }

        return new GenericUrl(expanded);
    }
}
