package com.perforce.hws.server;

import com.perforce.hws.p4base.P4JavaRuntimeException;
import com.perforce.hws.p4base.ServerHandle;
import com.perforce.hws.server.sessions.SessionData;
import com.perforce.p4java.PropertyDefs;
import com.perforce.p4java.exception.AccessException;
import com.perforce.p4java.exception.ConnectionException;
import com.perforce.p4java.exception.P4JavaException;
import com.perforce.p4java.impl.mapbased.rpc.RpcPropertyDefs;
import com.perforce.p4java.impl.mapbased.rpc.RpcServer;
import com.perforce.p4java.option.server.TrustOptions;
import com.perforce.p4java.server.IServer;
import com.perforce.p4java.server.ServerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Semaphore;
import org.apache.http.HttpStatus;

import static spark.Spark.halt;

/**
 * Created by tristan on 4/8/16.
 */
public interface UsesServerHandles {

    /** The logger. */
    Logger LOGGER = LoggerFactory.getLogger(UsesServerHandles.class);

    /** The server to locks. */
    // Yep. Static instance.
    ConcurrentMap<String, Semaphore> SERVER_TO_LOCKS = new ConcurrentHashMap<>();

    /**
     * The Interface ConsumesHandle.
     *
     * @param <Value> the generic type
     */
    @FunctionalInterface
    interface ConsumesHandle<Value> {

        /**
         * Apply.
         *
         * @param serverHandle the server handle
         * @return the value
         * @throws Exception the exception
         */
        Value apply(ServerHandle serverHandle) throws Exception;
    }

    /**
     * Executes a code block where we just generally wrap any problem
     * with a RuntimeException.
     *
     * @param <T> the generic type
     * @param serverId the server id
     * @param sessionData the session data
     * @param settings the settings
     * @param function the function
     * @return the t
     */
    default <T> T withServerHandle(String serverId,
                                   SessionData sessionData,
                                   HWSSettings settings,
                                   ConsumesHandle<T> function) {
        try (ServerHandle serverHandle = obtainServerHandle(
        		serverId, sessionData, settings)) {
            if (serverHandle == null) {
                halt(HttpStatus.SC_FORBIDDEN);
            }
            return function.apply(serverHandle);
        } catch (UnknownServerException e) {
        	halt(HttpStatus.SC_NOT_FOUND);
        }  catch (RuntimeException e) {
        	throw e;
        } catch (Exception e) {
        	throw new RuntimeException(e);
        }
        return null;
    }

    /**
     * Unlike withServerHandle there is no session on this server connection.
     * 
     * If we can not obtain a server handle, we do not call the function.
     *
     * @param <T> the generic type
     * @param serverId the server id
     * @param settings the settings
     * @param function the function
     * @return the t
     */
    default <T> T withSessionlessServerHandle(String serverId,
                                              HWSSettings settings,
                                              ConsumesHandle<T> function) {
        try (ServerHandle serverHandle = obtainServerHandle(serverId, settings)) {
            if (serverHandle == null) {
                return null;
            }
            return function.apply(serverHandle);
        } catch (UnknownServerException e) {
        	halt(HttpStatus.SC_NOT_FOUND);
        } catch (RuntimeException e) {
        	throw e;
        } catch (Exception e) {
        	throw new RuntimeException(e);
        }
        return null;
    }

    /**
     * Creates the server handle based on information we'll typically
     * have available in routes.
     * <p>
     * This is intended to be used in a try-with-resources block to deal
     * with connections. In fact,
     * you should pretty much always check for null, and associate
     * that with a 403, since we
     * probably do not have authorization. E.g.
     * <p>
     * try ( ServerHandle serverHandle = obtainServerHandle(...) ) {
     * if (serverHandle == null) {
     * halt(403);
     * }
     * <p>
     * // Continue....
     * }
     *
     * @param serverId    Our server ID to connect to.
     * @param sessionData We should have a registered session before continuing
     * @param settings    Application settings we pull most p4d settings from
     * @return the server handle
     */
    default ServerHandle obtainServerHandle(String serverId,
                                            SessionData sessionData,
                                            HWSSettings settings) {

        P4dConfig p4dConfig = settings.getP4dConfigMap().get(serverId);

        if (p4dConfig == null) {
            throw new UnknownServerException(serverId);
        }

        SessionData.P4LoginInfo loginInfo =
        		sessionData.getP4LoginInfoMap().get(serverId);
        if (loginInfo == null) {
            return null;
        }

        return obtainServerHandle(p4dConfig, loginInfo, settings);
    }

    /**
     * Special case of obtainServerHandle to be used when there is no user
     * session (e.g, when you're
     * logging into the server.
     *
     * @param serverId If we can't find server configuration for this id,
     * we'll return null.
     * @param settings Application settings (for server configuration values)
     * @return the server handle
     */
    default ServerHandle obtainServerHandle(String serverId, HWSSettings settings) {

        P4dConfig p4dConfig = settings.getP4dConfigMap().get(serverId);

        if (p4dConfig == null) {
        	throw new UnknownServerException(serverId);
        }

        return obtainServerHandle(p4dConfig, null, settings);
    }

    /**
     * Creates the ServerHandle instance. You probably want other methods
     * that check input arguments.
     *
     * @param p4dConfig The server configuration, assumed to not be null.
     * @param loginInfo Unless it's null, we'll pull the user and ticket from here.
     * @param settings the settings
     * @return the server handle
     */
    default ServerHandle obtainServerHandle(P4dConfig p4dConfig,
                                            SessionData.P4LoginInfo loginInfo,
                                            HWSSettings settings) {
        return new ServerHandle() {

            private IServer server;

            @Override
            public void close() {
                if (server != null) {
                    Semaphore semaphore = SERVER_TO_LOCKS.get(
                    		p4dConfig.toURIString());
                    semaphore.release();
                    if (server.isConnected()) {
                        try {
                            server.disconnect();
                        } catch (ConnectionException | AccessException e) {
                            LOGGER.debug("Ignoring problem during disconnect", e);
                        }
                    }
                    server = null;
                }
            }

            @Override
            public IServer get() {
                if (server != null) {
                    return server;
                }

                try {
                    Semaphore newSemaphore =
                    		new Semaphore(settings.getMaxServerConnections());
                    Semaphore semaphore = SERVER_TO_LOCKS.putIfAbsent(
                    		p4dConfig.toURIString(), newSemaphore);
                    // This is unintuitive behavior of putIfAbsent, but, if we've
                    // just associated the uri, it returns null (not the default
                    // value we've just set)
                    if (semaphore == null) {
                        semaphore = newSemaphore;
                    }
                    semaphore.acquire();

                    server = ServerFactory.getServer(
                    		p4dConfig.toURIString(), createP4JavaProperties());
                    if (p4dConfig.getP4CHARSET() != null) {
                        if ("none".equals(p4dConfig.getP4CHARSET())) {
                            server.setCharsetName(null);
                        } else {
                            server.setCharsetName(p4dConfig.getP4CHARSET());
                        }
                    }

                    if (loginInfo != null && loginInfo.getUser() != null) {
                        server.setUserName(loginInfo.getUser());
                    }

                    if (p4dConfig.getAPILEVEL() != null) {
                        int apiLevel = Integer.valueOf(p4dConfig.getAPILEVEL());
                        ((RpcServer) server).setClientApiLevel(apiLevel);
                    } else if (0 < settings.getDefaultApiLevel()) {
                        ((RpcServer) server).setClientApiLevel(
                        		settings.getDefaultApiLevel());
                    }

                    if (p4dConfig.isSsl()) {
                        RpcServer rpcServer = (RpcServer) server;
                        if (settings.getTrustFingerprints() != null) {
                            LOGGER.info("reading fingerprints file {}",
                            		settings.getTrustFingerprints());
                            TrustOptions opts = new TrustOptions(true, false, true);
                            Files.readAllLines(
                            	Paths.get(settings.getTrustFingerprints()))
                                    .stream()
                                    .filter(l -> !l.isEmpty())
                                    .forEach(fp -> {
                                        try {
                                            LOGGER.info(
                                            	"adding trust fingerprint {}", fp);
                                            rpcServer.addTrust(fp, opts);
                                        } catch (P4JavaException e) {
                                            throw new P4JavaRuntimeException(
                                        		"Unable to add trust: " + fp, e);
                                        }
                                    });
                        } else {
                            TrustOptions trustOptions = null;
                            if (settings.isEnableManInMiddleAttacks()) {
                                trustOptions = new TrustOptions(true, false, true);
                            } else if (settings.isAutoTrust()) {
                                trustOptions = new TrustOptions(false, false, true);
                            }
                            if (trustOptions != null) {
                                ((RpcServer) server).addTrust(
                                		new TrustOptions(true, false, true));
                            }
                        }
                    }

                    server.connect();

                    if (loginInfo != null && loginInfo.getTicket() != null) {
                        server.setAuthTicket(loginInfo.getTicket());
                    }

                    return server;
                } catch (URISyntaxException | P4JavaException e) {
                    throw new P4JavaRuntimeException(e);
                } catch (IOException e) {
                    throw new UncheckedIOException(e);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        };
    }

    /**
     * Creates the p4 java properties.
     *
     * @return the properties
     */
    default Properties createP4JavaProperties() {
        Properties p = new Properties();
        // Never ever touch local user files if possible. The server will need
        // to do this as well.
        //
        // The commons "ServerCOnnectionPoolFacade" also uses these properties:
        // - sockSoTimeout (RpcPropertyDefs.RPC_SOCKET_SO_TIMEOUT_NICK)
        p.setProperty("useAuthMemoryStore", "1");

        // Some commands are blocked by this check that are actually valid
        // commands.
        p.put(RpcPropertyDefs.RPC_RELAX_CMD_NAME_CHECKS_NICK, "true");

        p.put(PropertyDefs.PROG_NAME_KEY, "Helix Web Services");
        try {
            p.put(PropertyDefs.PROG_VERSION_KEY, VersionHelpers.productVersion());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }

        return p;
    }
}
