package com.perforce.hws.server;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Methods to help us grab our version string from the "Version" file that's
 * generated by the build process.
 */
public class VersionHelpers {

    /**
     * Tries to locate the Version file and return it's contents.
     *
     * If it can't locate it (likely because you're running in a development
     * mode) we'll return some default content.
     *
     * @return Contents of our version file.
     * @throws IOException
     */
    static public String versionContents() throws IOException {
        Path versionPath = Paths.get("Version");
        if (Files.exists(versionPath)) {
            return new String(Files.readAllBytes(versionPath));
        } else {
            return "RELEASE = 2099 9 main ;\n" +
                    "PATCHLEVEL = 9999999 ;\n" +
                    "SUPPDATE = 2099 09 09 ;";
        }
    }

    static final Pattern VERSION_FILE_KEY_VALUE = Pattern.compile("\\s*(?<key>\\S+)\\s*=\\s*(?<value>.*)\\s*;");

    /**
     * Parses the p4/Version file that is typically copied into our build or
     * distribution.
     *
     * @param content
     * @return
     */
    static public Map<String, String> parseVersionFile(String content) {
        return Arrays.stream(content.split("\n"))
                .map(VERSION_FILE_KEY_VALUE::matcher)
                .filter(Matcher::matches)
                .collect(Collectors.toMap(m -> m.group("key"), m -> m.group("value")));
    }

    /**
     * Create a standard version string from the Version file.
     *
     * @return The product version string to use
     */
    static public String productVersion() throws IOException {
        Map<String, String> versionInfo = parseVersionFile(versionContents());
        return "2016.1.0." + versionInfo.get("PATCHLEVEL");
    }
}
