package com.perforce.hws.server.filters;

import com.perforce.hws.server.*;
import org.apache.http.HttpStatus;
import spark.Filter;
import spark.Request;
import spark.Response;

import java.util.regex.Matcher;

import static spark.Spark.halt;

/**
 * Unless this is the "/api" request, looks for the platformVersion
 * property, and ensure's it's valid. If it is, we set the api level,
 * if it has not been set.
 */
public class PlatformVersionFilter implements Filter, ApiPathUtils {

    private HWSSettings settings;

    public PlatformVersionFilter(HWSSettings settings) {
        this.settings = settings;
    }

    @Override
    public void handle(final Request request, final Response response)
            throws Exception {
        if (request.pathInfo().equals(pathTo(ApiPath.Api)) ||
                request.pathInfo().equals(pathTo(ApiPath.Root))
                || request.pathInfo().startsWith(pathTo(ApiPath.Api, ApiPath.DocDir))
                || request.pathInfo().startsWith("/favicon.ico")
                || namespaceVersionPattern().matcher(request.pathInfo()).matches()) {
            return;
        }
        // Note: we have to pull this out of the request... manually,
        // since this is a filter.
        Matcher m = validPathPattern().matcher(request.pathInfo());
        if (!m.matches()) {
            halt(HttpStatus.SC_NOT_FOUND,
                    "Illegal path format: " + request.pathInfo());
        }

        String platformVersion = m.group("version");
        if (!PlatformVersions.isSupportedVersion(platformVersion)) {
            halt(HttpStatus.SC_NOT_FOUND, "Unsupported platform version");
        }

        PlatformVersion version = PlatformVersions.getVersion(platformVersion);
        request.attribute("platformVersion", version);

        HWSSettings hwsSettings = request.attribute("settings");

        if (hwsSettings.getDefaultApiLevel() == 0) {
            hwsSettings.setDefaultApiLevel(
                    Integer.valueOf(version.getDefaultApiLevel()));
        }
    }

    @Override
    public HWSSettings getSettings() {
        return settings;
    }
}
