package com.perforce.hws.server.routes;

import static spark.Spark.halt;

import java.time.Instant;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.perforce.hws.server.HWSSettings;
import com.perforce.hws.server.RequestHelper;
import com.perforce.hws.server.sessions.SessionData;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureException;
import spark.Request;
import spark.Response;
import spark.Route;

/**
 * A decorator to wrap a Spark route that should require
 * authentication. 
 */
// Ideally we would have used a Spark filter (before) to
// intercept the route but before filters do not get
// passed request parameters - only the routes do and
// we need to determine the server we are dealing with
// from the request (for p4d routes)
public final class AuthenticatedRoute implements Route {
	/**
     * The logger.
     */
    private static final Logger LOGGER =
            LoggerFactory.getLogger(AuthenticatedRoute.class);
    
	/** The wrapped route. */
	private Route wrappedRoute;
	
	/**
	 * Creates a route wrapping the intended target.
	 *
	 * @param wrappedRoute the wrapped route
	 * @return the authenticated route
	 */
	public static AuthenticatedRoute create(final Route wrappedRoute) {
		LOGGER.debug("Creating AuthenticatedRoute");
		return new AuthenticatedRoute(wrappedRoute);
	}

	/**
	 * Instantiates a new authenticated route.
	 * @param wrappedRoute the wrapped route
	 */
	private AuthenticatedRoute(final Route wrappedRoute) {
		this.wrappedRoute = wrappedRoute;
	}

	/* (non-Javadoc)
	 * @see spark.Route#handle(spark.Request, spark.Response)
	 */
	@Override
	public Object handle(final Request request,
						 final Response response) throws Exception {
		String authHeader = request.headers("Authorization");
        if (authHeader == null) {
            halt(HttpStatus.SC_FORBIDDEN, "Authorization required");
        }
        HWSSettings settings = request.attribute("settings");
        try {
        	// The authorization should be signed with the signing
        	// key so decode that first
            Jws<Claims> claimsJws = Jwts.parser()
                        .setSigningKey(settings.getJwtSigningKey())
                        .parseClaimsJws(authHeader);

            // Check for timeout if allowed
            if (0 < settings.getJwtTimeoutInSeconds()) {
                Instant expires = claimsJws.getBody().getExpiration().toInstant();
                if (expires.isBefore(Instant.now())) {
                    halt(HttpStatus.SC_FORBIDDEN, "Expired");
                }
            }
            // We expect there to be a server as a parameter (for
            // example from a P4D login request) or for there to
            // be an auth server default in Hws settings
            String p4AuthServerId = request.params("server");
            // If there is not a server id in the request we expect
            // the default to have been configured in Hws settings.
            if (StringUtils.isEmpty(p4AuthServerId)) {
            	p4AuthServerId = settings.getAuthP4d();
            	if (StringUtils.isEmpty(p4AuthServerId)) {
            		halt(HttpStatus.SC_BAD_REQUEST, "Unspecified login server");
            	}
            }
            String sessionId = claimsJws.getBody().getId();
            // Put some session data for the P4D credentials in the session.
            // Note that this used to be handle in an 'in memory'
            // cache but as we have the user and ticket we do not need
            // that now. However all the routes rely on the session data
            // so we'll build one here for them to access
            SessionData sessionData = new SessionData();
            // Once the signing has been decoded the authorization should be
            // base64 encoded version of user:p4ticket
            String[] credentials = RequestHelper.getCredentials(sessionId);
            SessionData.P4LoginInfo loginInfo = new SessionData.P4LoginInfo();
            loginInfo.setUser(credentials[0]);
            loginInfo.setTicket(credentials[1]);
            sessionData.getP4LoginInfoMap().put(p4AuthServerId, loginInfo);
            request.attribute("sessionData", sessionData);
        } catch (SignatureException sigEx) {
            halt(HttpStatus.SC_FORBIDDEN, "Invalid or illegal signature");
        }
		return this.wrappedRoute.handle(request, response);
	}

}
