package com.perforce.hws.server.routes;

import com.google.gson.JsonSyntaxException;
import com.perforce.hws.p4base.HasGson;

import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import spark.Request;

import static spark.Spark.halt;

/**
 * Nearly all requests are intended to be mapped to a single JSON model, so
 * apply some basic commonality via methods here.
 */
public interface UsesJsonRequests extends HasGson {

	/** Logger to record the request processing. */
    Logger LOGGER = LoggerFactory.getLogger(UsesJsonRequests.class);

    /**
     * Parse the request as a specific class type, halting with 400 error codes
     * if the user gave us no, or invalid JSON.
     *
     * @param request The Spark HTTP request
     * @param modelType Our requested model class to be converted by Gson
     * @param <Model> The model type
     * @return The model type, should generally never be null
     */
    default <Model> Model readModel(Request request, Class<Model> modelType) {
        String body = request.body();
        if (body == null || body.isEmpty()) {
        	LOGGER.warn("The request body was empty for"
        	    + request.pathInfo() + ", method=" + request.requestMethod());
            halt(HttpStatus.SC_BAD_REQUEST, "no request content");
        }
        try {
        	LOGGER.debug("Generating " + modelType + " data from json body");
            return gson.fromJson(body, modelType);
        } catch (JsonSyntaxException syntaxError) {
            LOGGER.warn("Illegal JSON " + body + " for " + modelType, syntaxError);
            halt(HttpStatus.SC_BAD_REQUEST, syntaxError.getMessage());
        }
        // Java doesn't detect that the halt method actually throws an exception.
        return null;
    }
}
