package com.perforce.hws.server.routes.perforce;

import static com.perforce.hws.server.ApiPath.Doc;
import static com.perforce.hws.server.VersionHelpers.productVersion;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.perforce.hws.server.ApiPathUtils;
import com.perforce.hws.server.HWSSettings;
import com.perforce.hws.server.PlatformVersions;

import spark.Request;
import spark.Response;
import spark.Route;

/**
 * Route to generate a simple version page for the P4d namespace.
 */
public class PerforceVersionRoute implements Route, ApiPathUtils {
	/** The gson. */
    private Gson gson;

    /** The settings. */
    private HWSSettings settings;

    /**
     * Instantiates a new perforce version route.
     *
     * @param settings the settings
     */
    public PerforceVersionRoute(final HWSSettings settings) {
        this.settings = settings;
    }

    /* (non-Javadoc)
     * @see spark.Route#handle(spark.Request, spark.Response)
     */
	@Override
    public Object handle(final Request request, final Response response)
    		throws Exception {
		String version = productVersion();
        List<String> supportedVersions =
        	Arrays.stream(PlatformVersions.SUPPORTED_VERSIONS)
                .map(v -> v.getName())
                .collect(Collectors.toList());

        request.attribute("response_content_type_set", "true");

        if (request.headers("Accept").contains("application/json")) {
        	gson = new GsonBuilder().
            		setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ").create();
            response.header("Content-Type", "application/json");
            Map<String, Object> versionInfo = new HashMap<>();
            versionInfo.put("productVersion", version);
            versionInfo.put("supportedVersions", supportedVersions);
            return gson.toJson(versionInfo);
        } else {
            response.header("Content-Type", "text/html");
            return "<html>"
                   + "<head><title>Helix Web Services</title></head>"
                   + "<body>"
                   + "<h1> Helix Web Services " + version + "</h1>"
                   + "<ul>"
                   + "<li><a href=\"" + apiPathTo(Doc)
                   + "\">User Guide</a></li>"
                   + "</ul>"
                   + "<h2> Supported versions </h2>"
                   + "<ul>"
                   + supportedVersions.stream().
                    	map(v -> "<li>" + v + "</li>")
                    		.collect(Collectors.joining())
                   + "</ul>"
                   + "</body>"
                   + "</html>";
        }
    }

    /* (non-Javadoc)
     * @see com.perforce.hws.server.ApiPathUtils#getSettings()
     */
    @Override
    public HWSSettings getSettings() {
        return settings;
    }
}
