package com.perforce.hws.server;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.testng.Assert;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import com.esotericsoftware.yamlbeans.YamlReader;
import com.perforce.hws.server.HWSSettings.WebHook;
import com.perforce.hws.server.HWSSettings.WhitelistedCommand;

/**
 * Unit testing for the HWSSettings.
 */
public class HWSSettingsTest {
	
	/** The settings. */
	private HWSSettings settings;
	
	/** Path to config test file. */
	private static final String SYSTEM_CONF_PATH =
			"src/test/resources/test-hws.conf";
	
	/**
	 * Runs before every test.
	 */
	@BeforeMethod
	public void before() {
		settings = new HWSSettings();
	}
	
	/**
	 * Test reset p4d config map.
	 */
	@Test(expectedExceptions = IllegalStateException.class)
	public void testResetP4dConfigMapNoDir() {
		settings.resetP4dConfigMap();
	}
	
	/**
	 * Assert set from config.
	 *
	 * @param config the config
	 * @param key the key
	 * @param supplier the supplier
	 */
	private void assertSetFromConfig(final Map<String, Object> config,
									 final ConfigurationKey key,
									 final Supplier<?> supplier) {
		Assert.assertEquals(config.get(key.toString()), supplier.get().toString());
	}
	
	/**
	 * Assert set from config bool.
	 *
	 * @param config the config
	 * @param key the key
	 * @param supplier the supplier
	 */
	private void assertSetFromConfigBool(final Map<String, Object> config,
			 final ConfigurationKey key,
			 final Supplier<Boolean> supplier) {
		Assert.assertEquals(
				Boolean.valueOf((String) config.get(key.toString())),
				supplier.get());
	}
	
	/**
	 * src/test/resources/p4ds contains 3 files.
	 * localhost - good configuration
	 * empty - empty file
	 * nonsense - badly formed Yaml
	 * Test that the result of building a map of P4ds contains
	 * just the good localhost entry.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testP4dConfig() throws IOException {
		settings.setSystemConfigPath(SYSTEM_CONF_PATH);
		settings.overrideFromSystemConfig();
		Map<String, P4dConfig> configMap = settings.getP4dConfigMap();
		Assert.assertTrue(configMap.keySet().size() == 1);
		Assert.assertTrue(configMap.containsKey("localhost"));
		Assert.assertEquals("localhost", configMap.get("localhost").getId());
		Assert.assertEquals("Local P4D used for testing",
							configMap.get("localhost").getDescription());
	}
	
	/**
	 * Test set from config file.
	 * @throws FileNotFoundException the file not found exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testOverrideFromSystemConfig()
			throws FileNotFoundException, IOException {
		settings.setSystemConfigPath(SYSTEM_CONF_PATH);
		settings.overrideFromSystemConfig();
		try (FileReader fileReader = new FileReader(SYSTEM_CONF_PATH)) {
            YamlReader yamlReader = new YamlReader(fileReader);
            @SuppressWarnings("unchecked")
			Map<String, Object> config = (Map<String, Object>) yamlReader.read();
            assertSetFromConfig(config,
            		ConfigurationKey.ACCESS_CONTROL_ALLOW_ORIGIN,
            		settings::getAccessControlAllowOrigin);
            assertSetFromConfig(config,
            		ConfigurationKey.ACCESS_CONTROL_ALLOW_HEADERS,
            		settings::getAccessControlAllowHeaders);
            assertSetFromConfig(config,
            		ConfigurationKey.ACCESS_CONTROL_REQUEST_METHOD,
            		settings::getAccessControlRequestMethod);
            assertSetFromConfigBool(config,
            		ConfigurationKey.AUTO_TRUST,
            		settings::isAutoTrust);
            assertSetFromConfig(config,
            		ConfigurationKey.DEFAULT_API_LEVEL,
            		settings::getDefaultApiLevel);
            assertSetFromConfigBool(config,
            		ConfigurationKey.ENABLE_GIT_FUSION,
            		settings::isEnableGitFusion);
            assertSetFromConfigBool(config,
            		ConfigurationKey.ENABLE_HTTPS,
            		settings::isEnableHttps);
            assertSetFromConfigBool(config,
            		ConfigurationKey.ENABLE_MAN_IN_MIDDLE_ATTACKS,
            		settings::isEnableManInMiddleAttacks);
            assertSetFromConfig(config,
            		ConfigurationKey.GITFUSIONDEPOT,
            		settings::getGitFusionDepot);
            assertSetFromConfig(config,
            		ConfigurationKey.HWS_AUTH_P4D,
            		settings::getAuthP4d);
            assertSetFromConfig(config,
            		ConfigurationKey.HWS_PORT,
            		settings::getHwsPort);
            assertSetFromConfig(config,
            		ConfigurationKey.JWT_SIGNING_KEY,
            		settings::getJwtSigningKey);
            assertSetFromConfig(config,
            		ConfigurationKey.JWT_TIMEOUT_IN_SECONDS,
            		settings::getJwtTimeoutInSeconds);
            assertSetFromConfig(config,
            		ConfigurationKey.KEYSTORE_FILE,
            		settings::getKeystoreFile);
            assertSetFromConfig(config,
            		ConfigurationKey.KEYSTORE_PASSWORD,
            		settings::getKeystorePassword);
            assertSetFromConfig(config,
            		ConfigurationKey.MAX_SERVER_CONNECTIONS,
            		settings::getMaxServerConnections);
            assertSetFromConfig(config,
            		ConfigurationKey.P4DCONFIGDIR,
            		settings::getP4dConfigDir);
            assertSetFromConfig(config,
            		ConfigurationKey.PREFIX,
            		settings::getPrefix);
            assertSetFromConfig(config,
            		ConfigurationKey.REQUEST_FILTER_PATH,
            		settings::getRequestFilterPath);
            assertSetFromConfig(config,
            		ConfigurationKey.SETUMASK,
            		settings::getSetumask);
            assertSetFromConfig(config,
            		ConfigurationKey.SETGID,
            		settings::getSetgid);
            assertSetFromConfig(config,
            		ConfigurationKey.SETUID,
            		settings::getSetuid);
            assertSetFromConfig(config,
            		ConfigurationKey.TRUSTSTORE_FILE,
            		settings::getTruststoreFile);
            assertSetFromConfig(config,
            		ConfigurationKey.TRUSTSTORE_PASSWORD,
            		settings::getTruststorePassword);
            assertSetFromConfig(config,
            		ConfigurationKey.TRUST_FINGERPRINTS,
            		settings::getTrustFingerprints);
            assertSetFromConfig(config,
            		ConfigurationKey.WORKSPACE_DIR,
            		settings::getWorkspaceDir);
            List<WebHook> webhooks = settings.getWebHooks();
            Assert.assertEquals("1", webhooks.get(0).getId());
            List<WhitelistedCommand> whiteList = settings.getCommandWhitelist();
            Assert.assertEquals("info", whiteList.get(0).getCommand());
            Assert.assertTrue(whiteList.get(0).getRequiredArgs().contains("xx"));
            Assert.assertTrue(whiteList.get(0).getRequiredArgs().contains("yy"));
		}
	}
}
