package com.perforce.workshop.tjuricek.initializer_gradle_plugin

import com.perforce.p4java.server.IOptionsServer
import com.perforce.workshop.tjuricek.p4java_ext.P4Connection
import com.perforce.workspace.tjuricek.initializer.Initializer
import org.gradle.api.DefaultTask
import org.gradle.api.file.FileTree
import org.gradle.api.tasks.TaskAction

import java.util.function.Supplier

/**
 */
class InitializePerforceTask extends DefaultTask {

    @TaskAction
    def initializePerforce() {

        P4Connection p4Connection = getP4Connection();

        try {
            Initializer initializer = new Initializer(getServerSupplier(p4Connection))
            initializer.seedPerforce(getPerforceOptions())

//            // After initialization, make sure the background perforce plugin
//            // knows what we set the super user to.
//            initializer.loadPerforceItems().users.first()
        } finally {
            p4Connection.close();
        }
    }

    private def getP4Connection() {
        return P4Connection.create(
                project.extensions.perforce.hostname,
                project.extensions.perforce.port,
                false,
                "initializer-gradle-plugin",
                "0.1.0"
        );
    }

    private def getServerSupplier(P4Connection p4Connection) {
        return new Supplier<IOptionsServer>() {
            @Override
            IOptionsServer get() {
                return p4Connection.getOptionsServer()
            }
        };
    }

    private def getPerforceOptions() {
        Initializer.PerforceOptions options = new Initializer.PerforceOptions();

        if (project.extensions.perforceInitializer.hasProperty("templateContext")) {
            options.setTemplateContext(project.extensions.perforceInitializer.templateContext);
        }

        options.setClientRootDir(getClientRootDir());
        options.setTriggersDir(getTriggersDir());
        options.setDefinitions(getDefinitions());
        options.setInputStreamLoader(getInputStreamLoader());

        return options;
    }

    private def getClientRootDir() {
        File clientRoot = project.file("${project.buildDir}/perforce/clients");
        if (project.extensions.perforceInitializer.hasProperty("clientRoot")) {
            clientRoot = project.extensions.perforceInitializer.clientRoot;
        }
        return clientRoot;
    }

    /**
     * Put triggers under the working directory of the perforce server.
     */
    private def getTriggersDir() {
        def startPerforceTask = project.tasks.getByName('startPerforce')
        return new File(startPerforceTask.workDir, "triggers")
    }

    /**
     * Provide input streams to json files in the src/test/perforce directory,
     * which can be overridden
     */
    private def getDefinitions() {

        FileTree defTree = project.fileTree(dir: sourceDir, include: '**/*.json')

        Set<File> files = defTree.files

        Iterator<File> fileIterator = files.iterator()

        return new Iterator<InputStream>() {
            @Override
            boolean hasNext() {
                return fileIterator.hasNext()
            }

            @Override
            InputStream next() {
                File nextFile = fileIterator.next()
                return new FileInputStream(nextFile)
            }
        }
    }

    /**
     * Try to load references relative to the source directory.
     */
    private def getInputStreamLoader() {
        File sourceDir = this.sourceDir
        return new Initializer.InputStreamLoader() {
            @Override
            InputStream getInputStream(String resourceName) {
                File resFile = new File(sourceDir, resourceName);
                return new FileInputStream(resFile);
            }
        }
    }

    def getSourceDir() {
        File defRoot = project.file("src/test/perforce")
        if (project.extensions.perforceInitializer.hasProperty("source")) {
            defRoot = project.extensions.perforceInitializer.source
        }
        return defRoot;
    }
}
